<?php

namespace Drupal\extend_help_maintainers;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;

/**
 * Builds a "Maintainers" render array for a module's help page.
 *
 * This service reads the `maintainers` key from a module's .info.yml file and
 * creates a themed render array displaying the maintainers' avatars, names,
 * and profile links. It also applies caching and attaches the required library.
 *
 * Example usage:
 * @code
 * $build = \Drupal::service('extend_help_maintainers.help_builder')
 *   ->build('extend_help_maintainers');
 * @endcode
 */
class MaintainersHelpBuilder {

  use StringTranslationTrait;

  /**
   * The module extension list service.
   *
   * Provides information about all installed modules, including paths and
   * .info.yml data.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected ModuleExtensionList $extension_list;

  /**
   * Constructs a MaintainersHelpBuilder object.
   *
   * @param \Drupal\Core\Extension\ModuleExtensionList $extension_list
   *   The module extension list service.
   */
  public function __construct(ModuleExtensionList $extension_list) {
    $this->extension_list = $extension_list;
  }

  /**
   * Builds a render array for a module's maintainers block.
   *
   * @param string $module_name
   *   The machine name of the module whose maintainers should be displayed.
   *
   * @return array|null
   *   A renderable array with the following keys:
   *   - #theme: The theme hook to use ('extend_help_maintainers').
   *   - #title: The block title ("Maintainers").
   *   - #maintainers: An array of maintainers (name, avatar, profile).
   *   - #placeholder: Default avatar URL if maintainer avatar is missing.
   *   - #weight: Block weight for ordering.
   *   - #attached: Libraries to attach.
   *   Returns NULL if there are no maintainers.
   *
   * @throws \Drupal\Core\Extension\Exception\UnknownExtensionException
   *   Thrown if the specified module does not exist.
   */
  public function build(string $module_name): ?array {
    // Get module info from .info.yml
    $info = $this->extension_list->getExtensionInfo($module_name);
    $maintainers = $info['maintainers'] ?? [];

    // Return null if no maintainers defined
    if (empty($maintainers)) {
      return NULL;
    }

    // Path to this module for default placeholder avatar
    $module_path = $this->extension_list
      ->getPath('extend_help_maintainers');
    $placeholder = Url::fromUri('base:' . $module_path . '/images/user-placeholder.svg')->toString();

    // Build render array
    $build = [
      '#theme' => 'extend_help_maintainers',
      '#title' => $this->t('Maintainers'),
      '#maintainers' => array_values(array_filter($maintainers, fn($m) => !empty($m['name']))),
      '#placeholder' => $placeholder,
      '#weight' => 100,
      '#attached' => [
        'library' => ['extend_help_maintainers/maintainers'],
      ],
    ];

    // Add caching metadata: cache for 1 day
    (new CacheableMetadata())
      ->setCacheMaxAge(86400)
      ->applyTo($build);

    return $build;
  }
}
