<?php

namespace Drupal\extend_help_maintainers\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\extend_help_maintainers\Plugin\MaintainersFetcher\MaintainersFetcherManager;

/**
 * Configuration form for Extend Help Maintainers.
 *
 * Allows administrators to select which fetcher plugins
 * should be used to display maintainers on help pages.
 */
class ExtendHelpMaintainersSettingsForm extends ConfigFormBase {

  /**
   * Config key for module settings.
   */
  public const SETTINGS = 'extend_help_maintainers.settings';

  /**
   * The MaintainersFetcher plugin manager.
   *
   * @var \Drupal\extend_help_maintainers\Plugin\MaintainersFetcher\MaintainersFetcherManager
   */
  protected MaintainersFetcherManager $fetcherManager;

  /**
   * Constructs the settings form.
   *
   * @param \Drupal\extend_help_maintainers\Plugin\MaintainersFetcher\MaintainersFetcherManager $fetcherManager
   *   The plugin manager for maintainers fetchers.
   */
  public function __construct(MaintainersFetcherManager $fetcherManager) {
    $this->fetcherManager = $fetcherManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create($container) {
    return new static(
      $container->get('extend_help_maintainers.maintainers_fetcher_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [self::SETTINGS];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'extend_help_maintainers_settings_form';
  }

  /**
   * Builds the configuration form.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form render array.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config(self::SETTINGS);

    // Collect all available fetcher plugins.
    $options = [];
    foreach ($this->fetcherManager->getDefinitions() as $plugin_id => $definition) {
      $options[$plugin_id] = $definition['label'] ?? $plugin_id;
    }

    $form['selected_plugins'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Preferred Maintainers Fetchers'),
      '#description' => $this->t('Select which fetcher plugins should be used to display maintainers.'),
      '#options' => $options,
      '#default_value' => $config->get('selected_plugins') ?? array_keys($options),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Handles form submission.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $selected = array_filter($form_state->getValue('selected_plugins'));

    $this->configFactory->getEditable(self::SETTINGS)
      ->set('selected_plugins', $selected)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
