<?php

namespace Drupal\extend_help_maintainers;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\extend_help_maintainers\Plugin\MaintainersFetcher\MaintainersFetcherManager;

/**
 * Builds a "Maintainers" render array for a module's help page.
 *
 * This service reads maintainers metadata and creates a themed render array
 * displaying avatars, names, and profile links. It also applies caching and
 * attaches the required library.
 *
 * Example usage:
 * @code
 * $build = \Drupal::service('extend_help_maintainers.help_builder')
 *   ->build('extend_help_maintainers', $maintainers);
 * @endcode
 */
class MaintainersHelpBuilder {

  use StringTranslationTrait;

  /**
   * Provides information about installed modules, including paths and .info.yml data.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected ModuleExtensionList $extension_list;

  /**
   * Plugin manager for fetching maintainers from different sources.
   *
   * @var \Drupal\extend_help_maintainers\Plugin\MaintainersFetcher\MaintainersFetcherManager
   */
  protected MaintainersFetcherManager $fetcherManager;

  /**
   * Constructs a MaintainersHelpBuilder object.
   *
   * @param \Drupal\Core\Extension\ModuleExtensionList $extension_list
   *   The module extension list service.
   * @param \Drupal\extend_help_maintainers\Plugin\MaintainersFetcher\MaintainersFetcherManager $fetcherManager
   *   The plugin manager for maintainers fetchers.
   */
  public function __construct(ModuleExtensionList $extension_list, MaintainersFetcherManager $fetcherManager) {
    $this->extension_list = $extension_list;
    $this->fetcherManager = $fetcherManager;
  }

  /**
   * Builds a render array for a module's maintainers block.
   *
   * @param string $module_name
   *   The machine name of the module whose maintainers should be displayed.
   * @param array $maintainers
   *   An array of maintainers (name, avatar, profile).
   *
   * @return array|null
   *   A render array with:
   *   - #theme: 'extend_help_maintainers'
   *   - #title: "Maintainers"
   *   - #maintainers: Array of maintainers
   *   - #placeholder: Default avatar URL
   *   - #attached: Libraries
   *   Returns NULL if no maintainers are defined.
   *
   * @throws \Drupal\Core\Extension\Exception\UnknownExtensionException
   *   Thrown if the specified module does not exist.
   */
  public function build(string $module_name, array $maintainers): ?array {
    if (empty($maintainers)) {
      return NULL;
    }

    // Path to this module for default placeholder avatar.
    $module_path = $this->extension_list->getPath('extend_help_maintainers');
    $placeholder = Url::fromUri(
      'base:' . $module_path . '/images/user-placeholder.svg',
      ['absolute' => TRUE]
    )->toString();

    // Build render array.
    $build = [
      '#theme' => 'extend_help_maintainers',
      '#title' => $this->t('Maintainers'),
      '#maintainers' => array_values(array_filter(
        $maintainers,
        fn($m) => !empty($m['name'])
      )),
      '#placeholder' => $placeholder,
      '#weight' => 100,
      '#attached' => [
        'library' => ['extend_help_maintainers/extend_help_maintainers.maintainers'],
      ],
    ];

    // Add caching metadata: cache for 1 day.
    (new CacheableMetadata())
      ->setCacheMaxAge(86400)
      ->setCacheContexts(['url.path', 'user.permissions', 'languages:language_interface'])
      ->setCacheTags(["module:$module_name"])
      ->applyTo($build);

    return $build;
  }

}
