<?php

namespace Drupal\extend_help_maintainers\Merge;

use Drupal\extend_help_maintainers\DTO\Maintainer;

/**
 * Service for deduplicating and merging maintainers.
 *
 * Ensures that maintainers with the same identifier (Drupal.org username
 * or name) are merged into a single DTO, preferring non-empty values.
 */
class MaintainersMerger {

  /**
   * Deduplicate and merge maintainers.
   *
   * The merger is order-independent: it always prefers non-empty values.
   *
   * @param \Drupal\extend_help_maintainers\DTO\Maintainer[] $items
   *   Array of maintainer DTOs.
   *
   * @return \Drupal\extend_help_maintainers\DTO\Maintainer[]
   *   Array of merged maintainer DTOs (re-indexed).
   */
  public function merge(array $items): array {
    $result = [];

    foreach ($items as $maintainer) {
      if (!$maintainer instanceof Maintainer) {
        continue;
      }

      $id = $maintainer->getIdentifier();

      if (!isset($result[$id])) {
        $result[$id] = $maintainer;
        continue;
      }

      $existing = $result[$id];

      $name = $this->prefer($existing->getName(), $maintainer->getName());
      $drupalOrg = $this->prefer($existing->getDrupalOrg(), $maintainer->getDrupalOrg());
      $avatar = $this->prefer($existing->getAvatar(), $maintainer->getAvatar());

      $result[$id] = new Maintainer(
        $name,
        $drupalOrg,
        $avatar
      );
    }

    // Return a flat array of merged maintainers.
    return array_values($result);
  }

  /**
   * Prefer a non-empty value.
   *
   * @param string|null $a
   *   First candidate value.
   * @param string|null $b
   *   Second candidate value.
   *
   * @return string|null
   *   The preferred non-empty value.
   */
  private function prefer(?string $a, ?string $b): ?string {
    return !empty($b) ? $b : $a;
  }

}
