<?php

namespace Drupal\extend_help_maintainers\Plugin\MaintainersFetcher;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\extend_help_maintainers\DTO\Maintainer;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Fetches maintainers from Drupal.org project page.
 *
 * @MaintainersFetcher(
 *   id = "drupal_org",
 *   label = @Translation("Drupal.org Maintainers Fetcher"),
 *   description = @Translation("Scrapes maintainers information from the Drupal.org project page."),
 *   priority = 10
 * )
 */
class DrupalOrgMaintainersFetcher extends PluginBase implements ContainerFactoryPluginInterface, MaintainersFetcherInterface {

  /**
   * HTTP client for external requests.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $http_client;

  /**
   * Constructs the plugin.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ClientInterface $http_client) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->http_client = $http_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('http_client')
    );
  }

  /**
   * Fetch maintainers from Drupal.org project page.
   *
   * @param string $module_name
   *   The machine name of the module.
   *
   * @return \Drupal\extend_help_maintainers\DTO\Maintainer[]
   *   Array of Maintainer DTOs.
   */
  public function fetchMaintainers(string $module_name): array {
    $cache_key = 'extend_help_maintainers:drupal_org:' . $module_name;
    $cache = \Drupal::cache()->get($cache_key);
    if ($cache) {
      return $cache->data;
    }

    $maintainers = [];
    $url = 'https://www.drupal.org/project/' . $module_name;

    try {
      $response = $this->http_client->get($url, ['timeout' => 5]);
      $html = (string) $response->getBody();

      $raw_maintainers = $this->parseMaintainersFromHtml($html);

      foreach ($raw_maintainers as $data) {
        $maintainers[] = new Maintainer(
          $data['name'] ?? '',
          $data['drupal_org'] ?? null,
          $data['avatar'] ?? null,
        );
      }
    }
    catch (\Throwable $e) {
      \Drupal::logger('extend_help_maintainers')->error('Failed to fetch maintainers for @module: @message', [
        '@module' => $module_name,
        '@message' => $e->getMessage(),
      ]);
    }

    \Drupal::cache()->set($cache_key, $maintainers, REQUEST_TIME + 86400, ['project:' . $module_name]);

    return $maintainers;
  }

  /**
   * Parses maintainers block from Drupal.org project page HTML.
   *
   * @param string $html
   *   Raw HTML of the project page.
   *
   * @return array
   *   Array of maintainers [name, drupal_org, avatar].
   */
  protected function parseMaintainersFromHtml(string $html): array {
    $maintainers = [];

    libxml_use_internal_errors(TRUE);

    $dom = new \DOMDocument();
    $dom->loadHTML($html);

    $xpath = new \DOMXPath($dom);

    $nodes = $xpath->query(
      '//div[@id="block-drupalorg-project-maintainers"]//div[contains(@class,"maintainer")]'
    );

    foreach ($nodes as $node) {
      $nameNode = $xpath->query('.//small/a', $node)->item(0);
      $name = $nameNode?->textContent;

      $profile = $nameNode?->getAttribute('href');
      if ($profile && str_starts_with($profile, '/')) {
        $profile = 'https://www.drupal.org' . $profile;
      }

      // Extract Drupal.org username from profile URL.
      $drupalOrg = null;
      if ($profile && preg_match('#/u/([^/]+)#', $profile, $matches)) {
        $drupalOrg = $matches[1];
      }

      $imgNode = $xpath->query('.//img', $node)->item(0);
      $avatar = $imgNode?->getAttribute('src');

      if ($name) {
        $maintainers[] = [
          'name' => trim($name),
          'drupal_org' => $drupalOrg,
          'avatar' => $avatar,
        ];
      }
    }

    libxml_clear_errors();

    return $maintainers;
  }

}
