<?php

namespace Drupal\extend_help_maintainers\Plugin\MaintainersFetcher;

use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\extend_help_maintainers\DTO\Maintainer;
use Drupal\extend_help_maintainers\Plugin\MaintainersFetcher\MaintainersFetcherInterface;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Fetch maintainers from a module's .info.yml file.
 *
 * @MaintainersFetcher(
 *   id = "info_maintainers",
 *   label = @Translation("Info YAML Maintainers"),
 *   description = @Translation("Reads maintainers metadata from a module's .info.yml file."),
 *   priority = 100
 * )
 */
class InfoMaintainersFetcher extends PluginBase implements ContainerFactoryPluginInterface, MaintainersFetcherInterface {

  /**
   * Provides information about installed modules.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected ModuleExtensionList $extensionList;

  /**
   * HTTP client for external requests.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $http_client;

  /**
   * Constructs the plugin.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ModuleExtensionList $extension_list, ClientInterface $http_client) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->extensionList = $extension_list;
    $this->http_client = $http_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('extension.list.module'),
      $container->get('http_client')
    );
  }

  /**
   * Fetch maintainers from a module's .info.yml file.
   *
   * @param string $module_name
   *   The machine name of the module.
   *
   * @return \Drupal\extend_help_maintainers\DTO\Maintainer[]
   *   Array of Maintainer DTOs.
   */
  public function fetchMaintainers(string $module_name): array {
    $info = $this->extensionList->getExtensionInfo($module_name);

    $maintainers_data = $info['extra']['extend_help_maintainers']['maintainers']
      ?? $info['maintainers']
      ?? [];

    $maintainers = [];

    foreach ($maintainers_data as $data) {
      if (empty($data['name'])) {
        continue;
      }

      $avatar = $data['avatar'] ?? null;
      if (!$avatar && !empty($data['drupal_org'])) {
        $avatar = $this->fetchAvatarFromDrupalOrg($data['drupal_org']);
      }

      $maintainers[] = new Maintainer(
        $data['name'],
        $data['drupal_org'] ?? NULL,
        $avatar,
      );
    }

    return $maintainers;
  }

  /**
   * Attempts to fetch a maintainer's avatar from Drupal.org profile page.
   *
   * @param string $drupalOrg
   *   The Drupal.org username.
   *
   * @return string|null
   *   The avatar URL, or NULL if not found.
   */
  protected function fetchAvatarFromDrupalOrg(string $drupalOrg): ?string {
    $url = 'https://www.drupal.org/u/' . $drupalOrg . '/';
    try {
      $response = $this->http_client->get($url, ['timeout' => 5]);
      $html = (string) $response->getBody();

      if (preg_match('#<div[^>]+class="user-picture"[^>]*>.*?<img[^>]+src="([^"]+)"#is', $html, $matches)) {
        return $matches[1];
      }
    }
    catch (\Exception $e) {
      \Drupal::logger('extend_help_maintainers')->error('Avatar fetch error for @user: @message', [
        '@user' => $drupalOrg,
        '@message' => $e->getMessage(),
      ]);
    }

    return null;
  }

}
