<?php

namespace Drupal\extend_help_maintainers\DTO;

/**
 * Data Transfer Object representing a module maintainer.
 *
 * Encapsulates maintainer metadata such as name, Drupal.org username,
 * and avatar URL. Provides helper methods for deduplication and
 * conversion to render-friendly arrays.
 */
class Maintainer {

  /**
   * Maintainer's display name.
   *
   * @var string
   */
  protected string $name;

  /**
   * Drupal.org username (machine name).
   *
   * @var string|null
   */
  protected ?string $drupalOrg;

  /**
   * Avatar image URL.
   *
   * @var string|null
   */
  protected ?string $avatar;

  /**
   * Constructs a Maintainer DTO.
   *
   * @param string $name
   *   The maintainer's name.
   * @param string|null $drupalOrg
   *   The Drupal.org username.
   * @param string|null $avatar
   *   The avatar image URL.
   */
  public function __construct(
    string $name,
    ?string $drupalOrg = NULL,
    ?string $avatar = NULL
  ) {
    $this->name = trim($name);
    $this->drupalOrg = $drupalOrg ? trim($drupalOrg) : NULL;
    $this->avatar = $avatar ?: NULL;
  }

  /**
   * Gets the maintainer's name.
   */
  public function getName(): string {
    return $this->name;
  }

  /**
   * Gets the Drupal.org username.
   */
  public function getDrupalOrg(): ?string {
    return $this->drupalOrg;
  }

  /**
   * Gets the avatar image URL.
   */
  public function getAvatar(): ?string {
    return $this->avatar;
  }

  /**
   * Returns a unique identifier for deduplication.
   *
   * Uses Drupal.org username if available, otherwise falls back to name.
   */
  public function getIdentifier(): string {
    return strtolower($this->drupalOrg ?? $this->name);
  }

  /**
   * Converts the DTO to a render-friendly array.
   *
   * @return array
   *   Associative array with keys:
   *   - name: Maintainer name
   *   - drupal_org: Drupal.org username
   *   - avatar: Avatar URL
   */
  public function toArray(): array {
    return [
      'name' => $this->name,
      'drupal_org' => $this->drupalOrg,
      'avatar' => $this->avatar,
    ];
  }

}
