<?php

namespace Drupal\extend_help_maintainers\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\extend_help_maintainers\Plugin\MaintainersFetcher\MaintainersFetcherManager;

/**
 * Configuration form for Extend Help Maintainers.
 *
 * Allows administrators to select which fetcher plugins
 * should be used to display maintainers on help pages.
 */
class ExtendHelpMaintainersSettingsForm extends ConfigFormBase {

  /**
   * Config key for module settings.
   */
  public const SETTINGS = 'extend_help_maintainers.settings';

  /**
   * The MaintainersFetcher plugin manager.
   *
   * @var \Drupal\extend_help_maintainers\Plugin\MaintainersFetcher\MaintainersFetcherManager
   */
  protected MaintainersFetcherManager $fetcherManager;

  /**
   * Constructs the settings form.
   *
   * @param \Drupal\extend_help_maintainers\Plugin\MaintainersFetcher\MaintainersFetcherManager $fetcherManager
   *   The plugin manager for maintainers fetchers.
   */
  public function __construct(MaintainersFetcherManager $fetcherManager) {
    $this->fetcherManager = $fetcherManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create($container) {
    return new static(
      $container->get('extend_help_maintainers.maintainers_fetcher_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [self::SETTINGS];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'extend_help_maintainers_settings_form';
  }

  /**
   * Builds the configuration form.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form render array.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config(self::SETTINGS);

    // Collect all available fetcher plugins with their priorities.
    $options = [];
    $priorities = [];
    $custom_priorities = $config->get('plugin_priorities') ?? [];

    foreach ($this->fetcherManager->getSortedDefinitions() as $plugin_id => $definition) {
      $label = $definition['label'] ?? $plugin_id;
      $default_priority = $definition['priority'] ?? 0;
      $current_priority = $custom_priorities[$plugin_id] ?? $default_priority;

      $options[$plugin_id] = $label;
      $priorities[$plugin_id] = [
        'default' => $default_priority,
        'current' => $current_priority,
        'description' => $definition['description'] ?? '',
      ];
    }

    $form['selected_plugins'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Enabled Maintainers Fetchers'),
      '#description' => $this->t('Select which fetcher plugins should be used to display maintainers. Plugins are processed in priority order (highest first).'),
      '#options' => $options,
      '#default_value' => $config->get('selected_plugins') ?? array_keys($options),
    ];

    // Priority configuration section.
    $form['priorities'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Source Priorities'),
      '#description' => $this->t('Configure priority for each source. Higher values take precedence when merging maintainers from multiple sources. Leave empty to use plugin default.'),
      '#tree' => TRUE,
    ];

    foreach ($priorities as $plugin_id => $info) {
      $form['priorities'][$plugin_id] = [
        '#type' => 'number',
        '#title' => $options[$plugin_id],
        '#description' => $this->t('Default: @default. @description', [
          '@default' => $info['default'],
          '@description' => $info['description'],
        ]),
        '#default_value' => $info['current'] !== $info['default'] ? $info['current'] : '',
        '#min' => 0,
        '#step' => 1,
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * Handles form submission.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $selected = array_filter($form_state->getValue('selected_plugins'));
    $priorities = $form_state->getValue('priorities', []);

    // Filter out empty priority values (use plugin defaults).
    $plugin_priorities = [];
    foreach ($priorities as $plugin_id => $priority) {
      if ($priority !== '' && $priority !== NULL) {
        $plugin_priorities[$plugin_id] = (int) $priority;
      }
    }

    $this->configFactory->getEditable(self::SETTINGS)
      ->set('selected_plugins', $selected)
      ->set('plugin_priorities', $plugin_priorities)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
