<?php

namespace Drupal\extend_help_maintainers;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;

/**
 * Builds a "Maintainers" render array for a module's help page.
 *
 * This service creates a themed render array displaying avatars, names,
 * and profile links. It works only with arrays and has no knowledge of DTOs.
 *
 * Example usage:
 * @code
 * $build = \Drupal::service('extend_help_maintainers.help_builder')
 *   ->build('extend_help_maintainers', $maintainers_array);
 * @endcode
 */
class MaintainersHelpBuilder {

  use StringTranslationTrait;

  /**
   * Provides information about installed modules, including paths and .info.yml data.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected ModuleExtensionList $extension_list;

  /**
   * Constructs a MaintainersHelpBuilder object.
   *
   * @param \Drupal\Core\Extension\ModuleExtensionList $extension_list
   *   The module extension list service.
   */
  public function __construct(ModuleExtensionList $extension_list) {
    $this->extension_list = $extension_list;
  }

  /**
   * Builds a render array for a module's maintainers block.
   *
   * @param string $module_name
   *   The machine name of the module whose maintainers should be displayed.
   * @param array $maintainers
   *   An array of maintainer arrays with keys: name, drupal_org, avatar.
   *
   * @return array|null
   *   A render array with:
   *   - #theme: 'extend_help_maintainers'
   *   - #title: "Maintainers"
   *   - #maintainers: Array of maintainers
   *   - #placeholder: Default avatar URL
   *   - #attached: Libraries
   *   Returns NULL if no maintainers are defined.
   *
   * @throws \Drupal\Core\Extension\Exception\UnknownExtensionException
   *   Thrown if the specified module does not exist.
   */
  public function build(string $module_name, array $maintainers): ?array {
    if (empty($maintainers)) {
      return NULL;
    }

    // Path to this module for default placeholder avatar.
    $module_path = $this->extension_list->getPath('extend_help_maintainers');
    $placeholder = Url::fromUri(
      'base:' . $module_path . '/images/user-placeholder.svg',
      ['absolute' => TRUE]
    )->toString();

    // Build render array with sanitized maintainers data.
    $sanitized_maintainers = [];
    foreach ($maintainers as $maintainer) {
      if (empty($maintainer['name'])) {
        continue;
      }

      // Build profile URL from drupal_org username if available.
      $profile = NULL;
      if (!empty($maintainer['drupal_org'])) {
        $profile = 'https://www.drupal.org/u/' . $maintainer['drupal_org'];
      }

      $sanitized_maintainers[] = [
        'name' => $maintainer['name'],
        'drupal_org' => $maintainer['drupal_org'] ?? NULL,
        'avatar' => $maintainer['avatar'] ?? NULL,
        'profile' => $profile,
      ];
    }

    if (empty($sanitized_maintainers)) {
      return NULL;
    }

    $build = [
      '#theme' => 'extend_help_maintainers',
      '#title' => $this->t('Maintainers'),
      '#maintainers' => $sanitized_maintainers,
      '#placeholder' => $placeholder,
      '#weight' => 100,
      '#attached' => [
        'library' => ['extend_help_maintainers/extend_help_maintainers.maintainers'],
      ],
    ];

    // Add caching metadata: cache for 1 day.
    (new CacheableMetadata())
      ->setCacheMaxAge(86400)
      ->setCacheContexts(['url.path', 'user.permissions', 'languages:language_interface'])
      ->setCacheTags(["module:$module_name"])
      ->applyTo($build);

    return $build;
  }

}
