<?php

namespace Drupal\extend_help_maintainers\Plugin\MaintainersFetcher;

use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Traversable;

/**
 * Plugin manager for MaintainersFetcher plugins.
 *
 * Responsible for discovering, instantiating, and sorting
 * plugins that provide maintainers metadata.
 */
class MaintainersFetcherManager extends DefaultPluginManager {

  /**
   * Constructs the plugin manager.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   Handles module hooks and info.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend for plugin definitions.
   * @param \Traversable $namespaces
   *   Namespaces to search for plugin implementations.
   */
  public function __construct(
    ModuleHandlerInterface $module_handler,
    CacheBackendInterface $cache_backend,
    Traversable $namespaces = NULL
  ) {
    // Fallback to container service if namespaces not provided.
    $namespaces = $namespaces ?? \Drupal::service('container.namespaces');

    parent::__construct(
      'Plugin/MaintainersFetcher',
      $namespaces,
      $module_handler,
      MaintainersFetcherInterface::class,
      'Drupal\extend_help_maintainers\Annotation\MaintainersFetcher'
    );

    $this->setCacheBackend($cache_backend, 'extend_help_maintainers_fetchers');
    $this->alterInfo('extend_help_maintainers_fetcher_info');
  }

  /**
   * Returns plugin definitions sorted by priority.
   *
   * @param bool $descending
   *   Whether to sort in descending order (default TRUE).
   *
   * @return array
   *   Sorted plugin definitions.
   */
  public function getSortedDefinitions(bool $descending = TRUE): array {
    $definitions = $this->getDefinitions();

    uasort($definitions, static function ($a, $b) use ($descending) {
      $result = ($a['priority'] ?? 0) <=> ($b['priority'] ?? 0);
      return $descending ? -$result : $result;
    });

    return $definitions;
  }

}
