<?php

namespace Drupal\Tests\extend_help_maintainers\Kernel;

use Drupal\KernelTests\KernelTestBase;

/**
 * Kernel tests for MaintainersService.
 *
 * @group extend_help_maintainers
 * @coversDefaultClass \Drupal\extend_help_maintainers\Service\MaintainersService
 */
class MaintainersServiceTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'extend_help_maintainers',
  ];

  /**
   * Tests service is registered and accessible.
   *
   * @covers ::buildMaintainersBlock
   */
  public function testServiceRegistered(): void {
    $service = $this->container->get('extend_help_maintainers.maintainers_service');

    $this->assertNotNull($service);
    $this->assertInstanceOf(
      'Drupal\extend_help_maintainers\Service\MaintainersService',
      $service
    );
  }

  /**
   * Tests service returns render array for module with maintainers.
   *
   * @covers ::buildMaintainersBlock
   */
  public function testServiceReturnsRenderArray(): void {
    // Configure to use info_maintainers plugin.
    $this->config('extend_help_maintainers.settings')
      ->set('selected_plugins', ['info_maintainers'])
      ->save();

    $service = $this->container->get('extend_help_maintainers.maintainers_service');
    $build = $service->buildMaintainersBlock('extend_help_maintainers');

    $this->assertIsArray($build);
    $this->assertEquals('extend_help_maintainers', $build['#theme']);
    $this->assertArrayHasKey('#maintainers', $build);
    $this->assertArrayHasKey('#title', $build);
    $this->assertArrayHasKey('#placeholder', $build);
    $this->assertArrayHasKey('#attached', $build);
  }

  /**
   * Tests disabled fetcher is ignored.
   *
   * @covers ::buildMaintainersBlock
   */
  public function testDisabledFetcherIsIgnored(): void {
    $this->config('extend_help_maintainers.settings')
      ->set('selected_plugins', [])
      ->save();

    $service = $this->container->get('extend_help_maintainers.maintainers_service');
    $build = $service->buildMaintainersBlock('extend_help_maintainers');

    $this->assertNull($build);
  }

  /**
   * Tests service returns NULL for module without maintainers.
   *
   * @covers ::buildMaintainersBlock
   */
  public function testServiceReturnsNullForNoMaintainers(): void {
    // Use a module that doesn't have maintainers in info.yml.
    $this->config('extend_help_maintainers.settings')
      ->set('selected_plugins', ['info_maintainers'])
      ->save();

    $service = $this->container->get('extend_help_maintainers.maintainers_service');
    $build = $service->buildMaintainersBlock('system');

    // System module doesn't have maintainers, so should return NULL.
    $this->assertNull($build);
  }

  /**
   * Tests plugin manager is accessible.
   */
  public function testPluginManagerRegistered(): void {
    $manager = $this->container->get('extend_help_maintainers.maintainers_fetcher_manager');

    $this->assertNotNull($manager);
    $definitions = $manager->getDefinitions();
    $this->assertNotEmpty($definitions);
    $this->assertArrayHasKey('info_maintainers', $definitions);
  }

  /**
   * Tests plugins are sorted by priority.
   */
  public function testPluginsSortedByPriority(): void {
    $manager = $this->container->get('extend_help_maintainers.maintainers_fetcher_manager');
    $sorted = $manager->getSortedDefinitions();

    $priorities = [];
    foreach ($sorted as $plugin_id => $definition) {
      $priorities[] = $definition['priority'] ?? 0;
    }

    // Check that priorities are in descending order.
    $sorted_priorities = $priorities;
    rsort($sorted_priorities);
    $this->assertEquals($sorted_priorities, $priorities);
  }

  /**
   * Tests custom priorities from config override plugin defaults.
   *
   * @covers ::buildMaintainersBlock
   */
  public function testCustomPrioritiesOverrideDefaults(): void {
    // Set custom priority for info_maintainers (default is 100).
    $this->config('extend_help_maintainers.settings')
      ->set('selected_plugins', ['info_maintainers', 'drupal_org'])
      ->set('plugin_priorities', [
        'info_maintainers' => 50,
        'drupal_org' => 200,
      ])
      ->save();

    $service = $this->container->get('extend_help_maintainers.maintainers_service');
    
    // Service should work with custom priorities.
    $build = $service->buildMaintainersBlock('extend_help_maintainers');
    
    // If build is not null, priorities are being used.
    // The actual merge result depends on what data is available.
    $this->assertTrue($build === NULL || is_array($build));
  }

  /**
   * Tests builder service is registered.
   */
  public function testBuilderServiceRegistered(): void {
    $builder = $this->container->get('extend_help_maintainers.help_builder');

    $this->assertNotNull($builder);
    $this->assertInstanceOf(
      'Drupal\extend_help_maintainers\MaintainersHelpBuilder',
      $builder
    );
  }

  /**
   * Tests merger service is registered.
   */
  public function testMergerServiceRegistered(): void {
    $merger = $this->container->get('extend_help_maintainers.merger');

    $this->assertNotNull($merger);
    $this->assertInstanceOf(
      'Drupal\extend_help_maintainers\Merge\MaintainersMerger',
      $merger
    );
  }

}

