<?php

namespace Drupal\Tests\extend_help_maintainers\Unit;

use Drupal\extend_help_maintainers\DTO\Maintainer;
use PHPUnit\Framework\TestCase;

/**
 * Unit tests for Maintainer DTO.
 *
 * @group extend_help_maintainers
 * @coversDefaultClass \Drupal\extend_help_maintainers\DTO\Maintainer
 */
class MaintainerTest extends TestCase {

  /**
   * Tests identifier uses Drupal.org username when available.
   *
   * @covers ::getIdentifier
   */
  public function testIdentifierByDrupalOrg(): void {
    $m = new Maintainer('John Doe', 'johndoe');

    $this->assertEquals('johndoe', $m->getIdentifier());
  }

  /**
   * Tests identifier falls back to name when Drupal.org username is missing.
   *
   * @covers ::getIdentifier
   */
  public function testIdentifierFallbackToName(): void {
    $m = new Maintainer('John Doe');

    $this->assertEquals('john doe', $m->getIdentifier());
  }

  /**
   * Tests identifier is case-insensitive.
   *
   * @covers ::getIdentifier
   */
  public function testIdentifierCaseInsensitive(): void {
    $m1 = new Maintainer('John Doe', 'JohnDoe');
    $m2 = new Maintainer('Jane Doe', 'johndoe');

    $this->assertEquals('johndoe', $m1->getIdentifier());
    $this->assertEquals('johndoe', $m2->getIdentifier());
    $this->assertEquals($m1->getIdentifier(), $m2->getIdentifier());
  }

  /**
   * Tests toArray conversion.
   *
   * @covers ::toArray
   */
  public function testToArray(): void {
    $m = new Maintainer('John', 'john', 'avatar.png');

    $this->assertEquals([
      'name' => 'John',
      'drupal_org' => 'john',
      'avatar' => 'avatar.png',
    ], $m->toArray());
  }

  /**
   * Tests toArray with null values.
   *
   * @covers ::toArray
   */
  public function testToArrayWithNulls(): void {
    $m = new Maintainer('John Doe');

    $this->assertEquals([
      'name' => 'John Doe',
      'drupal_org' => NULL,
      'avatar' => NULL,
    ], $m->toArray());
  }

  /**
   * Tests getter methods.
   *
   * @covers ::getName
   * @covers ::getDrupalOrg
   * @covers ::getAvatar
   */
  public function testGetters(): void {
    $m = new Maintainer('John Doe', 'johndoe', 'avatar.png');

    $this->assertEquals('John Doe', $m->getName());
    $this->assertEquals('johndoe', $m->getDrupalOrg());
    $this->assertEquals('avatar.png', $m->getAvatar());
  }

  /**
   * Tests name trimming in constructor.
   *
   * @covers ::__construct
   */
  public function testNameTrimming(): void {
    $m = new Maintainer('  John Doe  ', '  johndoe  ');

    $this->assertEquals('John Doe', $m->getName());
    $this->assertEquals('johndoe', $m->getDrupalOrg());
  }

}

