<?php

namespace Drupal\Tests\extend_help_maintainers\Unit;

use Drupal\extend_help_maintainers\DTO\Maintainer;
use Drupal\extend_help_maintainers\Merge\MaintainersMerger;
use PHPUnit\Framework\TestCase;

/**
 * Unit tests for MaintainersMerger.
 *
 * @group extend_help_maintainers
 * @coversDefaultClass \Drupal\extend_help_maintainers\Merge\MaintainersMerger
 */
class MaintainersMergerTest extends TestCase {

  /**
   * Tests higher priority overrides fields.
   *
   * @covers ::merge
   */
  public function testHigherPriorityOverridesFields(): void {
    $low = new Maintainer('John', 'john', NULL);
    $high = new Maintainer('John', 'john', 'avatar.png');

    $merger = new MaintainersMerger();
    $result = $merger->merge([
      ['maintainer' => $low, 'priority' => 10],
      ['maintainer' => $high, 'priority' => 100],
    ]);

    $this->assertCount(1, $result);
    $merged = reset($result);
    $this->assertEquals('avatar.png', $merged->getAvatar());
    $this->assertEquals('John', $merged->getName());
    $this->assertEquals('john', $merged->getDrupalOrg());
  }

  /**
   * Tests lower priority fills gaps in higher priority.
   *
   * @covers ::merge
   */
  public function testLowerPriorityFillsGaps(): void {
    $high = new Maintainer('John', 'john', NULL);
    $low = new Maintainer('John', 'john', 'avatar.png');

    $merger = new MaintainersMerger();
    $result = $merger->merge([
      ['maintainer' => $high, 'priority' => 100],
      ['maintainer' => $low, 'priority' => 10],
    ]);

    $this->assertCount(1, $result);
    $merged = reset($result);
    // Higher priority name should be kept, but avatar from lower fills gap.
    $this->assertEquals('John', $merged->getName());
    $this->assertEquals('avatar.png', $merged->getAvatar());
  }

  /**
   * Tests equal priorities prefer non-empty values.
   *
   * @covers ::merge
   */
  public function testEqualPrioritiesPreferNonEmpty(): void {
    $m1 = new Maintainer('John', 'john', NULL);
    $m2 = new Maintainer('John', 'john', 'avatar.png');

    $merger = new MaintainersMerger();
    $result = $merger->merge([
      ['maintainer' => $m1, 'priority' => 50],
      ['maintainer' => $m2, 'priority' => 50],
    ]);

    $this->assertCount(1, $result);
    $merged = reset($result);
    $this->assertEquals('avatar.png', $merged->getAvatar());
  }

  /**
   * Tests deduplication by identifier.
   *
   * @covers ::merge
   */
  public function testDeduplicationByIdentifier(): void {
    $m1 = new Maintainer('John Doe', 'johndoe');
    $m2 = new Maintainer('John', 'johndoe');
    $m3 = new Maintainer('Jane', 'janedoe');

    $merger = new MaintainersMerger();
    $result = $merger->merge([
      ['maintainer' => $m1, 'priority' => 100],
      ['maintainer' => $m2, 'priority' => 50],
      ['maintainer' => $m3, 'priority' => 100],
    ]);

    $this->assertCount(2, $result);
    $identifiers = array_map(fn(Maintainer $m) => $m->getIdentifier(), $result);
    $this->assertContains('johndoe', $identifiers);
    $this->assertContains('janedoe', $identifiers);
  }

  /**
   * Tests backward compatibility with old format (Maintainer objects).
   *
   * @covers ::merge
   */
  public function testBackwardCompatibility(): void {
    $m1 = new Maintainer('John', 'john');
    $m2 = new Maintainer('Jane', 'jane');

    $merger = new MaintainersMerger();
    $result = $merger->merge([$m1, $m2]);

    $this->assertCount(2, $result);
  }

  /**
   * Tests merging multiple fields with priorities.
   *
   * @covers ::merge
   */
  public function testComplexMerge(): void {
    // High priority has name and drupal_org, but no avatar.
    $high = new Maintainer('John Doe', 'johndoe', NULL);
    // Low priority has avatar and different name.
    $low = new Maintainer('John', 'johndoe', 'avatar.png');

    $merger = new MaintainersMerger();
    $result = $merger->merge([
      ['maintainer' => $high, 'priority' => 100],
      ['maintainer' => $low, 'priority' => 10],
    ]);

    $this->assertCount(1, $result);
    $merged = reset($result);
    // High priority name should be kept.
    $this->assertEquals('John Doe', $merged->getName());
    // High priority drupal_org should be kept.
    $this->assertEquals('johndoe', $merged->getDrupalOrg());
    // Avatar from low priority should fill the gap.
    $this->assertEquals('avatar.png', $merged->getAvatar());
  }

  /**
   * Tests empty array returns empty result.
   *
   * @covers ::merge
   */
  public function testEmptyArray(): void {
    $merger = new MaintainersMerger();
    $result = $merger->merge([]);

    $this->assertIsArray($result);
    $this->assertCount(0, $result);
  }

  /**
   * Tests invalid items are skipped.
   *
   * @covers ::merge
   */
  public function testInvalidItemsSkipped(): void {
    $valid = new Maintainer('John', 'john');
    $invalid = 'not a maintainer';

    $merger = new MaintainersMerger();
    $result = $merger->merge([
      ['maintainer' => $valid, 'priority' => 100],
      $invalid,
    ]);

    $this->assertCount(1, $result);
    $this->assertEquals('John', reset($result)->getName());
  }

}

