# Contributing to Extend Help Maintainers

Thank you for your interest in contributing to Extend Help Maintainers! This document provides guidelines and instructions for contributing to the project.

## Code of Conduct

* Be respectful and inclusive
* Welcome newcomers and help them learn
* Focus on constructive feedback
* Follow Drupal's Code of Conduct

## How to Contribute

### Reporting Issues

Before reporting an issue:

1. Check if the issue already exists in the issue queue
2. Search the documentation and README
3. Test with the latest version of the module

When reporting an issue, please include:

* Drupal version
* Module version
* Steps to reproduce
* Expected vs. actual behavior
* Relevant error messages or logs
* Screenshots (if applicable)

### Submitting Patches

1. **Fork the repository** (if using Git)
2. **Create a feature branch** from the latest development branch
3. **Make your changes** following the coding standards below
4. **Write or update tests** for your changes
5. **Update documentation** if needed
6. **Test your changes** thoroughly
7. **Submit a patch** or pull request with a clear description

### Code Standards

#### Drupal Coding Standards

* Follow [Drupal Coding Standards](https://www.drupal.org/docs/develop/standards)
* Use [DrupalPractice](https://www.drupal.org/docs/develop/standards/coding-standards) standards
* Run code style checks before submitting:

```bash
./vendor/bin/phpcs --standard=Drupal,DrupalPractice src/ tests/
```

#### PHP Standards

* Follow PSR-12 coding standard
* Use strict type declarations where applicable
* Add proper type hints and return types
* Write self-documenting code with meaningful names

#### Code Structure

* Follow the existing architecture patterns:
  * Plugin-based for data sources
  * Service-oriented design
  * DTO pattern for data transfer
  * Dependency Injection throughout
* Maintain separation of concerns
* Keep classes focused and single-purpose

### Testing Requirements

All contributions must include tests:

#### Unit Tests

* Required for:
  * DTO classes
  * Pure logic (merge strategy, calculations)
  * Utility functions
* Located in `tests/src/Unit/`
* Must pass: `./vendor/bin/phpunit --testsuite=unit`

#### Kernel Tests

* Required for:
  * Service integration
  * Plugin discovery and instantiation
  * Configuration handling
  * Render array validation
* Located in `tests/src/Kernel/`
* Must pass: `./vendor/bin/phpunit --testsuite=kernel`

#### Test Guidelines

* Write descriptive test method names: `testHigherPriorityOverridesFields()`
* Test both success and failure cases
* Test edge cases (empty arrays, null values, etc.)
* Use `@covers` annotations to document what is tested
* Keep tests isolated and independent

### Documentation

#### Code Documentation

* Add PHPDoc comments for all public methods
* Document parameters and return types
* Include usage examples in docblocks where helpful
* Explain complex logic with inline comments

#### User Documentation

* Update README.md if adding features
* Update CHANGELOG.md for user-facing changes
* Add examples for new configuration options
* Update architecture diagram if structure changes

### Development Setup

1. **Clone the repository**:
   ```bash
   git clone [repository-url]
   cd extend_help_maintainers
   ```

2. **Install dependencies**:
   ```bash
   composer install
   ```

3. **Run tests**:
   ```bash
   ./vendor/bin/phpunit
   ```

4. **Check code style**:
   ```bash
   ./vendor/bin/phpcs --standard=Drupal,DrupalPractice src/ tests/
   ```

5. **Fix code style issues** (if any):
   ```bash
   ./vendor/bin/phpcbf --standard=Drupal,DrupalPractice src/ tests/
   ```

### Adding New Fetcher Plugins

To add a new maintainer data source:

1. **Create plugin class** in `src/Plugin/MaintainersFetcher/`
2. **Implement `MaintainersFetcherInterface`**
3. **Add annotation** with:
   - `id`: Unique plugin ID
   - `label`: Human-readable label
   - `description`: What the plugin does
   - `priority`: Default priority (higher = more important)
4. **Return array of `Maintainer` DTO objects** from `fetchMaintainers()`
5. **Add tests** for the new plugin
6. **Update documentation** if needed

Example:

```php
/**
 * @MaintainersFetcher(
 *   id = "custom_source",
 *   label = @Translation("Custom Source"),
 *   description = @Translation("Fetches maintainers from custom source."),
 *   priority = 50
 * )
 */
class CustomSourceFetcher extends PluginBase implements MaintainersFetcherInterface {
  // Implementation
}
```

### Commit Messages

Follow [Drupal commit message standards](https://www.drupal.org/docs/develop/git/using-git-to-contribute-to-drupal/writing-good-commit-messages):

* Use present tense: "Add feature" not "Added feature"
* First line: Brief summary (max 72 characters)
* Body: Explain what and why (if needed)
* Reference issue numbers: "Issue #1234567"

Examples:

```
Add validation for module names in hook_help_alter()

Prevents potential security issues by validating module names
before processing. Uses regex to ensure only valid module names
are accepted.

Issue #1234567
```

### Pull Request Guidelines

When submitting a pull request:

1. **Clear title** describing the change
2. **Detailed description**:
   - What changes were made
   - Why the changes were needed
   - How to test the changes
   - Screenshots (for UI changes)
3. **Reference related issues**
4. **Ensure all tests pass**
5. **Ensure code style is correct**
6. **Update CHANGELOG.md** if applicable

### Review Process

* All contributions will be reviewed
* Reviewers may request changes
* Be open to feedback and suggestions
* Respond to review comments promptly
* Update your contribution based on feedback

### Getting Help

* Check the [README.md](README.md) for documentation
* Review existing code for examples
* Ask questions in the issue queue
* Check Drupal.org documentation

## Architecture Guidelines

### Plugin System

* Plugins should be stateless where possible
* Use dependency injection for services
* Handle errors gracefully (log and return empty array)
* Don't throw exceptions that break the entire process

### Service Layer

* Services should orchestrate, not contain business logic
* Use interfaces for testability
* Keep services focused on their responsibility

### Data Transfer Objects (DTOs)

* DTOs should be immutable
* Use value objects for maintainers
* Provide clear conversion methods (`toArray()`)

### Testing Strategy

* Test behavior, not implementation
* Use mocks sparingly (prefer real objects in Kernel tests)
* Test edge cases and error conditions
* Maintain high test coverage

## Release Process

* Version numbers follow [Semantic Versioning](https://semver.org/)
* Major releases: Breaking changes
* Minor releases: New features (backward compatible)
* Patch releases: Bug fixes (backward compatible)

## License

By contributing, you agree that your contributions will be licensed under the GPL-2.0-or-later license, the same license as the project.

## Questions?

If you have questions about contributing:

* Open an issue in the issue queue
* Check the documentation in README.md
* Review existing code and tests for examples

Thank you for contributing to Extend Help Maintainers! 🎉

