<?php

namespace Drupal\extend_help_maintainers\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\extend_help_maintainers\MaintainersHelpBuilder;
use Drupal\extend_help_maintainers\DTO\Maintainer;
use Drupal\extend_help_maintainers\Merge\MaintainersMerger;
use Drupal\extend_help_maintainers\Plugin\MaintainersFetcher\MaintainersFetcherManager;
use Psr\Log\LoggerInterface;

/**
 * Service responsible for building the maintainers block
 * displayed on module help pages.
 *
 * This service orchestrates:
 * - Fetching maintainers via plugins.
 * - Merging duplicate/overlapping data.
 * - Passing normalized data to the HelpBuilder.
 */
class MaintainersService {

  /**
   * Help builder service.
   *
   * @var \Drupal\extend_help_maintainers\MaintainersHelpBuilder
   */
  protected MaintainersHelpBuilder $builder;

  /**
   * Plugin manager for maintainers fetchers.
   *
   * @var \Drupal\extend_help_maintainers\Plugin\MaintainersFetcher\MaintainersFetcherManager
   */
  protected MaintainersFetcherManager $fetcherManager;

  /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Merger service for combining maintainers data.
   *
   * @var \Drupal\extend_help_maintainers\Merge\MaintainersMerger
   */
  protected MaintainersMerger $merger;

  /**
   * Logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Constructs the service.
   */
  public function __construct(
    MaintainersHelpBuilder $builder,
    MaintainersFetcherManager $fetcher_manager,
    ConfigFactoryInterface $config_factory,
    MaintainersMerger $merger,
    LoggerInterface $logger
  ) {
    $this->builder = $builder;
    $this->fetcherManager = $fetcher_manager;
    $this->configFactory = $config_factory;
    $this->merger = $merger;
    $this->logger = $logger;
  }

  /**
   * Builds the maintainers block for a given module.
   *
   * @param string $module_name
   *   The machine name of the module.
   *
   * @return array|null
   *   A render array representing the maintainers block,
   *   or NULL if no maintainers were found.
   */
  public function buildMaintainersBlock(string $module_name): ?array {
    // Validate module name format.
    if (!preg_match('/^[a-z0-9_]+$/', $module_name)) {
      return NULL;
    }

    $config = $this->configFactory->get('extend_help_maintainers.settings');
    $selected_plugins = $config->get('selected_plugins') ?? [];

    $collected = [];

    // Get custom priorities from config (allows overriding plugin defaults).
    $custom_priorities = $config->get('plugin_priorities') ?? [];

    // Process plugins in priority order (highest first).
    foreach ($this->fetcherManager->getSortedDefinitions() as $plugin_id => $definition) {
      if (!in_array($plugin_id, $selected_plugins, TRUE)) {
        continue;
      }

      try {
        /** @var \Drupal\extend_help_maintainers\Plugin\MaintainersFetcher\MaintainersFetcherInterface $plugin */
        $plugin = $this->fetcherManager->createInstance($plugin_id);

        // Get priority: custom config override > plugin definition > default 0.
        $priority = $custom_priorities[$plugin_id] ?? $definition['priority'] ?? 0;

        foreach ($plugin->fetchMaintainers($module_name) as $maintainer) {
          // Pass maintainer with its source priority for proper merging.
          $collected[] = [
            'maintainer' => $maintainer,
            'priority' => $priority,
          ];
        }
      }
      catch (\Throwable $e) {
        // Log and skip failing plugin.
        $this->logger->error(
          'Failed to fetch maintainers from plugin @plugin: @message',
          ['@plugin' => $plugin_id, '@message' => $e->getMessage()]
        );
      }
    }

    /** @var Maintainer[] $merged */
    $merged = $this->merger->merge($collected);

    $maintainers_array = array_map(
      fn(Maintainer $m) => $m->toArray(),
      array_values($merged)
    );

    return $this->builder->build($module_name, $maintainers_array);
  }

}
