# Extend Help Maintainers

## Overview

**Extend Help Maintainers** is a developer-oriented Drupal module that enhances module help pages by displaying structured maintainer information from multiple sources.

The module introduces a pluggable architecture that allows maintainers to be fetched from different providers (e.g. module `.info.yml`, drupal.org, external APIs) and merged deterministically based on configurable priorities.

---

## Features

* 📦 Plugin-based maintainer providers (Fetcher plugins)
* 🧩 Support for multiple data sources
* 🧠 Deterministic merge strategy with priorities
* 🧱 Strongly-typed DTO (Value Object) for maintainers
* ⚙️ Configurable source selection
* 🧪 Fully testable architecture (Unit + Kernel tests)
* 🎨 Themed render output with caching support

---

## Use Case

Drupal core and contrib modules often contain maintainer information scattered across different places:

* `.info.yml` files
* drupal.org project metadata
* external systems

This module provides a **single, extensible system** to collect, normalize, merge, and display that information on module help pages.

---

## Architecture Overview

The module follows a layered architecture with clear separation of concerns:

```
┌────────────────────┐
│  MaintainersService│  ← Application / Facade layer
└─────────┬──────────┘
          │
          ▼
┌────────────────────┐
│ MaintainersFetcher │  ← Plugin Manager
│     Manager        │
└─────────┬──────────┘
          │
   ┌──────┴────────┐
   ▼               ▼
Info YAML     Drupal.org
Fetcher       Fetcher
(Plugin)      (Plugin)
   │               │
   └──────┬────────┘
          ▼
┌────────────────────┐
│ MaintainersMerger  │  ← Merge strategy (priority-based)
└─────────┬──────────┘
          ▼
┌────────────────────┐
│ MaintainersHelp    │  ← Render array builder
│     Builder        │
└────────────────────┘
```

For a detailed component diagram, see [`docs/architecture.puml`](docs/architecture.puml) (PlantUML format).

---

## Maintainer Data Model (DTO)

Each maintainer is represented by an immutable Value Object:

```php
Maintainer {
  name: string
  drupalOrg: ?string
  avatar: ?string
}
```

This guarantees:

* consistent structure
* predictable merging
* easy deduplication
* strong contracts between layers

---

## Fetcher Plugins

Maintainer data sources are implemented as plugins.

### Example: Info YAML Fetcher

```php
/**
 * @MaintainersFetcher(
 *   id = "info_maintainers",
 *   label = @Translation("Info YAML Maintainers"),
 *   priority = 100
 * )
 */
```

### Priority Rules

* Higher `priority` value wins
* Higher priority sources overwrite lower priority fields
* Deduplication is performed using a stable identifier

---

## Configuration

The module provides a configuration form where administrators can:

* Enable or disable specific fetcher plugins
* Control which data sources are used

Configuration is stored in:

```
extend_help_maintainers.settings
```

---

## Output

The module builds a themed render array:

* Displays maintainer avatars
* Links to drupal.org profiles
* Applies caching metadata
* Attaches required libraries

If no maintainers are available, nothing is rendered.

---

## Caching

The render output includes:

* Cache max-age: 24 hours
* Cache contexts: `url.path`, `user.permissions`
* Cache tags: `module:{module_name}`

This ensures correct invalidation while keeping rendering efficient.

---

## Extending the Module

### Adding a New Maintainer Source

1. Create a new Fetcher plugin
2. Implement `MaintainersFetcherInterface`
3. Return `Maintainer` DTO objects
4. Assign a priority

No changes to existing code are required.

---

## Installation

Install the module using Drush:

```bash
drush en extend_help_maintainers
```

Or via the Extend page at `/admin/modules`.

---

## Usage

1. Navigate to `/admin/config/system/extend-help-maintainers`
2. Select which fetcher plugins should be enabled
3. Optionally configure custom priorities for each source
4. Visit any module's help page (e.g., `/admin/help/system`) to see maintainers

The module automatically appends maintainers information to help pages via `hook_help_alter()`.

---

## Testing

The module includes comprehensive test coverage:

### Unit Tests

* DTO behavior and immutability
* Merge strategy logic with priorities
* Edge cases and error handling

Run unit tests:

```bash
./vendor/bin/phpunit --testsuite=unit
```

### Kernel Tests

* Service orchestration
* Plugin discovery and instantiation
* Configuration handling
* Render array validation
* Integration with Drupal core

Run kernel tests:

```bash
./vendor/bin/phpunit --testsuite=kernel
```

Run all tests:

```bash
./vendor/bin/phpunit
```

Test structure:

```
tests/
 └── src/
     ├── Unit/
     │   ├── MaintainerTest.php
     │   └── MaintainersMergerTest.php
     └── Kernel/
         └── MaintainersServiceTest.php
```

---

## Design Principles

* SOLID
* Separation of concerns
* Dependency Injection
* Plugin-driven extensibility
* Drupal coding standards

---

## Status

This module is intended for:

* Drupal 10+
* Developers and maintainers
* Reusable architectural patterns

---

## Maintainers

This module supports declaring maintainers directly in `.info.yml`:

```yaml
extra:
  extend_help_maintainers:
    maintainers:
      - name: Dmytro Porokhnya
        drupal_org: darkdim
        avatar: https://www.drupal.org/files/styles/grid-2-2x-square/public/user-pictures/picture-411912-1454223916.jpg
```

---

---

## CI/CD

This project uses GitLab CI/CD for automated testing and code quality checks:

* **Code style validation** - PHPCS checks on every merge request
* **Unit tests** - Automated unit test execution
* **Kernel tests** - Integration tests with Drupal
* **Security scanning** - Dependency and code security checks
* **Package building** - Automatic package creation for releases

See `.gitlab-ci.yml` for full pipeline configuration.

## Contributing

Contributions are welcome! Please see [CONTRIBUTING.md](CONTRIBUTING.md) for detailed guidelines.

Quick checklist:
* **Code style**: PSR-12 and Drupal Coding Standards
* **Tests**: All new features must include Unit and/or Kernel tests
* **Documentation**: Update README and inline documentation
* **Architecture**: Follow existing plugin-based patterns
* **Issues**: Report bugs and feature requests via the issue queue

For detailed contribution guidelines, including:
* How to set up development environment
* Testing requirements
* Code standards
* How to add new fetcher plugins
* Commit message format

See [CONTRIBUTING.md](CONTRIBUTING.md).

### Development Setup

1. Clone the repository
2. Install dependencies: `composer install`
3. Run tests: `./vendor/bin/phpunit`
4. Check code style: `./vendor/bin/phpcs --standard=Drupal,DrupalPractice src/ tests/`

---

## Support

* **Issues**: Report bugs and feature requests on [GitHub](https://github.com/your-username/extend_help_maintainers/issues)
* **Documentation**: See [`docs/architecture.puml`](docs/architecture.puml) for detailed architecture
* **Drupal.org**: Project page at `https://www.drupal.org/project/extend_help_maintainers`

---

## Release Notes

### 1.0.0

* Initial release
* Plugin-based architecture with Info YAML and Drupal.org fetchers
* Priority-based merge strategy
* Configurable source selection
* Full test coverage (Unit + Kernel)
* `hook_help_alter()` integration
* Caching support

---

## License

GPL-2.0-or-later

See [LICENSE.txt](LICENSE.txt) for full license text.

