<?php

namespace Drupal\extend_help_maintainers\Merge;

use Drupal\extend_help_maintainers\DTO\Maintainer;

/**
 * Service for deduplicating and merging maintainers from multiple sources.
 *
 * Merges maintainers with the same identifier (Drupal.org username or name)
 * into a single DTO, respecting source priorities. Higher priority sources
 * override lower priority ones, but non-empty values are always preferred
 * over empty ones regardless of priority.
 */
class MaintainersMerger {

  /**
   * Deduplicate and merge maintainers with priority support.
   *
   * @param array $items
   *   Array of items, where each item is either:
   *   - A Maintainer DTO (for backward compatibility, priority = 0)
   *   - An array with keys: 'maintainer' (Maintainer) and 'priority' (int)
   *
   * @return \Drupal\extend_help_maintainers\DTO\Maintainer[]
   *   Array of merged maintainer DTOs (re-indexed).
   */
  public function merge(array $items): array {
    $result = [];
    $priorities = [];

    foreach ($items as $item) {
      // Support both old format (Maintainer) and new format (array).
      if ($item instanceof Maintainer) {
        $maintainer = $item;
        $priority = 0;
      }
      elseif (is_array($item) && isset($item['maintainer']) && $item['maintainer'] instanceof Maintainer) {
        $maintainer = $item['maintainer'];
        $priority = $item['priority'] ?? 0;
      }
      else {
        continue;
      }

      $id = $maintainer->getIdentifier();

      if (!isset($result[$id])) {
        // First occurrence of this maintainer.
        $result[$id] = $maintainer;
        $priorities[$id] = $priority;
        continue;
      }

      // Merge with existing maintainer, respecting priorities.
      $existing = $result[$id];
      $existingPriority = $priorities[$id];

      // If priorities are equal, prefer non-empty values.
      if ($priority === $existingPriority) {
        $name = $this->preferNonEmpty($existing->getName(), $maintainer->getName());
        $drupalOrg = $this->preferNonEmpty($existing->getDrupalOrg(), $maintainer->getDrupalOrg());
        $avatar = $this->preferNonEmpty($existing->getAvatar(), $maintainer->getAvatar());
      }
      elseif ($priority > $existingPriority) {
        // New maintainer has higher priority - use it as base, but fill gaps from existing.
        $name = $this->preferNonEmpty($maintainer->getName(), $existing->getName());
        $drupalOrg = $this->preferNonEmpty($maintainer->getDrupalOrg(), $existing->getDrupalOrg());
        $avatar = $this->preferNonEmpty($maintainer->getAvatar(), $existing->getAvatar());
        $priorities[$id] = $priority;
      }
      else {
        // Existing maintainer has higher priority - use it as base, but fill gaps from new.
        $name = $this->preferNonEmpty($existing->getName(), $maintainer->getName());
        $drupalOrg = $this->preferNonEmpty($existing->getDrupalOrg(), $maintainer->getDrupalOrg());
        $avatar = $this->preferNonEmpty($existing->getAvatar(), $maintainer->getAvatar());
      }

      $result[$id] = new Maintainer($name, $drupalOrg, $avatar);
    }

    // Return a flat array of merged maintainers.
    return array_values($result);
  }

  /**
   * Prefer a non-empty value over an empty one.
   *
   * @param string|null $a
   *   First candidate value.
   * @param string|null $b
   *   Second candidate value.
   *
   * @return string|null
   *   The preferred non-empty value, or the first value if both are empty.
   */
  private function preferNonEmpty(?string $a, ?string $b): ?string {
    return !empty($b) ? $b : $a;
  }

}
