<?php

namespace Drupal\xntt_views\Plugin\views\data;

use Drupal\Core\Entity\EntityHandlerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\external_entities\ExternalEntityStorageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\TypedData\DataDefinitionInterface;

/**
 * Provides Views data for ExternalEntity.
 */
class ExternalEntityViewsData implements EntityHandlerInterface {

  use StringTranslationTrait;

  /**
   * Entity type for this views data handler instance.
   *
   * @var \Drupal\Core\Entity\EntityTypeInterface
   */
  protected $entityType;

  /**
   * The storage used for this entity type.
   *
   * @var \Drupal\external_entities\ExternalEntityStorageInterface
   */
  protected $storage;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The translation manager.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface
   */
  protected $translationManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * Constructs an EntityViewsData object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type to provide views integration for.
   * @param \Drupal\external_entities\ExternalEntityStorageInterface $storage_controller
   *   The storage handler used for this entity type.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $translation_manager
   *   The translation manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   */
  public function __construct(
    EntityTypeInterface $entity_type,
    ExternalEntityStorageInterface $storage_controller,
    EntityTypeManagerInterface $entity_type_manager,
    ModuleHandlerInterface $module_handler,
    TranslationInterface $translation_manager,
    EntityFieldManagerInterface $entity_field_manager,
  ) {
    $this->entityType = $entity_type;
    $this->entityTypeManager = $entity_type_manager;
    $this->storage = $storage_controller;
    $this->moduleHandler = $module_handler;
    $this->setStringTranslation($translation_manager);
    $this->entityFieldManager = $entity_field_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(
    ContainerInterface $container,
    EntityTypeInterface $entity_type,
  ) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('entity_type.manager'),
      $container->get('module_handler'),
      $container->get('string_translation'),
      $container->get('entity_field.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getViewsData(): array {
    $data = [];
    $xntt_id = $this->entityType->id();
    $xntt_label = $this->entityType->getLabel();
    $view_id = 'xntt_views_' . $xntt_id;

    // Base data.
    $data[$view_id] = [
      'table' => [
        'group' => $xntt_label,
        'provider' => $this->entityType->getProvider(),
        'base' => [
          'title' => $xntt_label,
          'help' => $this->t(
            '@xntttype external entity views mapping.',
            ['@xnttype' => $xntt_label]
          ),
          'query_id' => 'xntt_query',
          'xntt' => $xntt_id,
        ],
        'entity revision' => FALSE,
        'entity type' => $xntt_id,
      ],
      'rendered_entity' => [
        'field' => [
          'title' => $this->t('Rendered entity'),
          'help' => $this->t('Renders an entity in a view mode.'),
          'id' => 'rendered_external_entity',
        ],
      ],
    ];

    // Fields.
    $fields = $this->entityFieldManager->getFieldDefinitions(
      $xntt_id,
      $xntt_id
    );
    foreach ($fields as $field_id => $field_definition) {
      /** @var \Drupal\Core\Field\FieldStorageDefinitionInterface */
      $field_storage_definition = $field_definition->getFieldStorageDefinition();
      $main_property = $field_storage_definition->getMainPropertyName() ?? '';
      $propertie_names = $field_storage_definition->getPropertyNames();
      foreach ($propertie_names as $property_name) {
        $field_property_id = $field_id . '_' . $property_name;
        /** @var \Drupal\Core\TypedData\DataDefinitionInterface */
        $property_definition = $field_storage_definition
          ->getPropertyDefinition($property_name);

        if ($main_property === $property_name) {
          $field_property_id = $field_id;
          $data[$view_id][$field_id] = $this->mapFieldViewsData(
            $field_id,
            $field_definition,
            $property_name,
            $property_definition
          );
        }
        else {
          $data[$view_id][$field_property_id] = $this->mapFieldViewsData(
            $field_id,
            $field_definition,
            $property_name,
            $property_definition
          );
          $data[$view_id][$field_property_id]['title'] =
          $field_definition->getLabel() . ' (' . $property_name . ')';
        }
        $data[$view_id][$field_property_id]['entity field'] = $field_property_id;
      }
    }

    return $data;
  }

  /**
   * Provides the views data for a given field.
   *
   * Inspired from \Drupal\views\EntityViewsData::mapSingleFieldViewsData().
   *
   * @param string $field_id
   *   The field ID.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   * @param string $property_name
   *   The field property name to work on.
   * @param \Drupal\Core\TypedData\DataDefinitionInterface $property_definition
   *   The property definition.
   *
   * @return array
   *   The views data field definition.
   */
  protected function mapFieldViewsData(
    string $field_id,
    FieldDefinitionInterface $field_definition,
    string $property_name,
    DataDefinitionInterface $property_definition,
  ) {
    $views_field = [];
    $views_field['title'] = $field_definition->getLabel();
    if ($description = $field_definition->getDescription()) {
      $views_field['help'] = $description;
    }
    $field_type = $field_definition->getType();
    $property_type = $property_definition->getDataType();

    // Set up the field, sort, argument, and filters, based on
    // the column and/or field data type.
    switch ($field_type) {
      // Special case a few field types.
      case 'timestamp':
      case 'created':
      case 'changed':
        $views_field['field']['id'] = 'external_entity_field';
        $views_field['argument']['id'] = 'date';
        $views_field['filter']['id'] = 'date';
        $views_field['sort']['id'] = 'date';
        break;

      case 'language':
        $views_field['field']['id'] = 'external_entity_field_language';
        $views_field['argument']['id'] = 'language';
        $views_field['filter']['id'] = 'language';
        $views_field['sort']['id'] = 'standard';
        break;

      case 'boolean':
        $views_field['field']['id'] = 'external_entity_field';
        $views_field['argument']['id'] = 'numeric';
        $views_field['filter']['id'] = 'boolean';
        $views_field['sort']['id'] = 'standard';
        break;

      case 'uri':
        // Let's render URIs as URIs by default, not links.
        $views_field['field']['id'] = 'external_entity_field';
        $views_field['field']['default_formatter'] = 'string';

        $views_field['argument']['id'] = 'string';
        $views_field['filter']['id'] = 'string';
        $views_field['sort']['id'] = 'standard';
        break;

      case 'text':
      case 'text_with_summary':
        // Treat these three long text fields the same.
        $field_type = 'text_long';
        // Intentional fall-through here to the default processing!
      default:
        switch ($property_type) {
          case 'decimal':
          case 'float':
          case 'integer':
            $views_field['field']['id'] = 'external_entity_field';
            $views_field['argument']['id'] = 'numeric';
            $views_field['filter']['id'] = 'numeric';
            $views_field['sort']['id'] = 'standard';
            break;

          case 'email':
          case 'string':
          case 'uri':
            $views_field['field']['id'] = 'external_entity_field';
            $views_field['argument']['id'] = 'string';
            $views_field['filter']['id'] = 'string';
            $views_field['sort']['id'] = 'standard';
            break;

          case 'list':
          case 'timespan':
          case 'boolean':
          case 'timestamp':
          case 'language_reference':
          case 'datetime_iso8601':
          case 'duration_iso8601':
          case 'binary':
          case 'map':
          case 'language':
          case 'any':
          default:
            $views_field['field']['id'] = 'external_entity_field';
            $views_field['argument']['id'] = 'standard';
            $views_field['filter']['id'] = 'standard';
            $views_field['sort']['id'] = 'standard';
        }
    }

    if (!$field_definition->isRequired()) {
      // Provides "Is empty (NULL)" and "Is not empty (NOT NULL)" operators.
      $views_field['filter']['allow empty'] = TRUE;
    }

    // @todo Maybe add some post-processing (processViewsDataForLanguage,
    //   processViewsDataForEntityReference, processViewsDataForTextLong,
    //   processViewsDataForUuid).
    // @code
    // // Do post-processing for a few field types.
    // $process_method = 'processViewsDataFor'
    //   . Container::camelize($field_type);
    // if (method_exists($this, $process_method)) {
    //   $this->{$process_method}(
    //     $field_definition,
    //     $views_field, $column_name
    //   );
    // }
    // @endcode
    return $views_field;
  }

  /**
   * Processes the views data for a language field.
   *
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   * @param array $views_field
   *   The views field data.
   * @param string $field_column_name
   *   The field column being processed.
   */
  protected function processViewsDataForLanguage(FieldDefinitionInterface $field_definition, array &$views_field, $field_column_name) {
    // Apply special titles for the langcode field.
    if ($field_definition->getName() == $this->entityType->getKey('langcode')) {
      if ($table == $this->entityType->getDataTable() || $table == $this->entityType->getRevisionDataTable()) {
        $views_field['title'] = $this->t('Translation language');
      }
      if ($table == $this->entityType->getBaseTable() || $table == $this->entityType->getRevisionTable()) {
        $views_field['title'] = $this->t('Original language');
      }
    }
  }

  /**
   * Processes the views data for an entity reference field.
   *
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   * @param array $views_field
   *   The views field data.
   * @param string $field_column_name
   *   The field column being processed.
   */
  protected function processViewsDataForEntityReference(FieldDefinitionInterface $field_definition, array &$views_field, $field_column_name) {
    if ($entity_type_id = $field_definition->getItemDefinition()->getSetting('target_type')) {
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
      if ($entity_type instanceof ContentEntityType) {
        $views_field['relationship'] = [
          'base' => $this->getViewsTableForEntityType($entity_type),
          'base field' => $entity_type->getKey('id'),
          'label' => $entity_type->getLabel(),
          'title' => $entity_type->getLabel(),
          'id' => 'standard',
        ];
        $views_field['field']['id'] = 'external_entity_field';
        // Provide an argument plugin that has a meaningful titleQuery()
        // implementation getting the entity label.
        $views_field['argument']['id'] = 'entity_target_id';
        $views_field['argument']['target_entity_type_id'] = $entity_type_id;
        $views_field['filter']['id'] = 'numeric';
        $views_field['sort']['id'] = 'standard';
      }
      else {
        $views_field['field']['id'] = 'external_entity_field';
        $views_field['argument']['id'] = 'string';
        $views_field['filter']['id'] = 'string';
        $views_field['sort']['id'] = 'standard';
      }
    }

    if ($field_definition->getName() == $this->entityType->getKey('bundle')) {
      $views_field['filter']['id'] = 'bundle';
    }
  }

  /**
   * Processes the views data for a text field with formatting.
   *
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   * @param array $views_field
   *   The views field data.
   * @param string $field_column_name
   *   The field column being processed.
   */
  protected function processViewsDataForTextLong(FieldDefinitionInterface $field_definition, array &$views_field, $field_column_name) {
    // Connect the text field to its formatter.
    if ($field_column_name == 'value') {
      $views_field['field']['format'] = $field_definition->getName() . '__format';
      $views_field['field']['id'] = 'external_entity_field';
    }
  }

  /**
   * Processes the views data for a UUID field.
   *
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   * @param array $views_field
   *   The views field data.
   * @param string $field_column_name
   *   The field column being processed.
   */
  protected function processViewsDataForUuid(FieldDefinitionInterface $field_definition, array &$views_field, $field_column_name) {
    // It does not make sense for UUID fields to be click sortable.
    $views_field['field']['click sortable'] = FALSE;
  }

}
