<?php

namespace Drupal\xntt_views\Plugin\views\field;

use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;

/**
 * Generic Views field handler for ExternalEntity fields (non-SQL).
 *
 * Provides a simple, robust rendering strategy:
 * - expects the query plugin to populate $row->_entity with a loaded entity,
 * - handles entity translations using the view language when available,
 * - returns a simple string rendering of the field value (first item / 'value'
 *   property if present). Use a custom handler if you need full formatter
 *   output.
 *
 * @ViewsField("external_entity_field")
 */
class ExternalEntityField extends FieldPluginBase {

  /**
   * No SQL query building for external entities.
   */
  public function query() {
    // Set alias as current field name for clickSort().
    $this->field_alias =
      $this->options['entity_field']
      ?? $this->definition['entity field']
      ?? 'unknown';
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    // Row must contain the loaded entity under _entity.
    if (empty($values->_entity)) {
      return ['#markup' => ''];
    }
    $entity = $values->_entity;

    // Determine requested field name from options/definition.
    $field_name = $this->options['entity_field'] ?? ($this->definition['entity field'] ?? NULL);

    // If no field specified, default to "not available".
    if (empty($field_name)) {
      return ['#markup' => $this->t('n/a')];
    }
    // Handle translation: prefer view langcode if set.
    $langcode = NULL;
    $rendering_language = $this->view->getDisplay()->getOption('rendering_language');
    if ($rendering_language === '***LANGUAGE_site_default***') {
      // Use the site default language.
      $langcode = \Drupal::languageManager()->getDefaultLanguage()->getId();
    }
    elseif ($rendering_language === '***LANGUAGE_language_interface***') {
      // Interface text language selected for page.
      $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();
    }
    elseif ($rendering_language === '***LANGUAGE_entity_translation***') {
      // Content language of view row.
      $langcode = $entity->language()->getId();
    }
    elseif ($rendering_language === '***LANGUAGE_entity_default***') {
      // Original language of content in view row.
      $langcode = $entity->getUntranslated()->language()->getId();
    }
    elseif (!str_starts_with('***LANGUAGE_', $rendering_language)) {
      // Specific language (e.g., 'fr').
      $langcode = $rendering_language;
    }
    else {
      // Fallback to current language.
      $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();
      // Warn for unsupported option.
      \Drupal::logger('xntt_views')->warning(
        'Unsupported rendering_language option "%option" in view %view_name. Falling back to current language "%langcode".',
        [
          '%option' => $rendering_language,
          '%view_name' => $this->view->id() ?? 'n/a',
          '%langcode' => $langcode,
        ]
      );
    }
    if ($entity->isTranslatable()
      && $langcode
      && $entity->hasTranslation($langcode)
    ) {
      $entity = $entity->getTranslation($langcode);
    }

    // Handle common property shortcuts.
    $entity_definition = $entity->getEntityType();
    $label_key = $entity_definition->getKey('label') ?: 'label';
    if ($field_name === $label_key) {
      return ['#markup' => (string) $entity->label()];
    }

    $id_key = $entity_definition->getKey('id') ?: 'id';
    if ($field_name === $id_key) {
      return ['#markup' => (string) $entity->id()];
    }

    // If the field exists as a Field API field, try to extract usable values.
    if ($entity->hasField($field_name)) {
      $field = $entity->get($field_name);
      if ($field->isEmpty()) {
        return ['#markup' => ''];
      }

      $items = [];
      foreach ($field as $item) {
        // Prefer scalar 'value' when present.
        if (isset($item->value)) {
          $items[] = ['#markup' => (string) $item->value];
          continue;
        }

        // If the item is renderable or castable, try that.
        try {
          $string = (string) $item;
          $items[] = ['#markup' => $string];
          continue;
        }
        catch (\Throwable $e) {
          // Fallback to exported value.
        }

        try {
          $items[] = ['#markup' => (string) print_r($item->getValue(), TRUE)];
        }
        catch (\Throwable $e) {
          $items[] = ['#markup' => ''];
        }
      }

      // If only one value, return it directly (render array or markup).
      if (count($items) === 1) {
        return $items[0];
      }

      // Multiple values: return an item list so Drupal renders them all.
      return [
        '#theme' => 'item_list',
        '#items' => array_map(
          function ($i) {
            // Items are already render arrays like ['#markup' => '...'].
            return $i;
          },
          $items
        ),
      ];
    }

    // Fallback to a public property or method on the entity.
    if (isset($entity->{$field_name})) {
      return ['#markup' => (string) $entity->{$field_name}];
    }
    if (method_exists($entity, $field_name)) {
      try {
        return ['#markup' => (string) $entity->{$field_name}()];
      }
      catch (\Throwable $e) {
        // Ignore and fallback below.
      }
    }

    // Nothing found: empty output.
    return ['#markup' => ''];
  }

  /**
   * Define an option for the handler to carry the field name.
   */
  public function defineOptions() {
    $options = parent::defineOptions();
    $options['entity_field'] = ['default' => NULL];
    return $options;
  }

}
