<?php

namespace Drupal\xntt_views\Plugin\views\field;

use Drupal\views\ResultRow;
use Drupal\views\Plugin\views\field\FieldPluginBase;

/**
 * Provides a Views field handler for entity language for External Entities.
 *
 * Shows the language of the entity (or of the translation used by the view if
 * a view language is set and the entity has that translation).
 *
 * @ViewsField("external_entity_field_language")
 */
class ExternalEntityFieldLanguage extends FieldPluginBase {

  /**
   * No SQL query building for external entities.
   */
  public function query() {
    // Set alias as current field name for clickSort().
    $this->field_alias =
      $this->options['entity_field']
      ?? $this->definition['entity field']
      ?? 'unknown';
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    if (empty($values->_entity)) {
      return ['#markup' => ''];
    }

    $entity = $values->_entity;

    // Determine candidate langcode:
    // - Prefer the view langcode when the entity is translatable and has that
    //   translation.
    // - Otherwise fall back to the entity inline language.
    $langcode = NULL;
    $rendering_language = $this->view->getDisplay()->getOption('rendering_language');
    if ($rendering_language === '***LANGUAGE_site_default***') {
      // Use the site default language.
      $langcode = \Drupal::languageManager()->getDefaultLanguage()->getId();
    }
    elseif ($rendering_language === '***LANGUAGE_language_interface***') {
      // Interface text language selected for page.
      $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();
    }
    elseif ($rendering_language === '***LANGUAGE_entity_translation***') {
      // Content language of view row.
      $langcode = $entity->language()->getId();
    }
    elseif ($rendering_language === '***LANGUAGE_entity_default***') {
      // Original language of content in view row.
      $langcode = $entity->getUntranslated()->language()->getId();
    }
    elseif (!str_starts_with('***LANGUAGE_', $rendering_language)) {
      // Specific language (e.g., 'fr').
      $langcode = $rendering_language;
    }
    else {
      // Fallback to entity language.
      $langcode = $entity->language()->getId();
      // Warn for unsupported option.
      \Drupal::logger('xntt_views')->warning(
        'Unsupported rendering_language option "%option" in view %view_name. Falling back to current language "%langcode".',
        [
          '%option' => $rendering_language,
          '%view_name' => $this->view->id() ?? 'n/a',
          '%langcode' => $langcode,
        ]
      );
    }

    if (empty($langcode)) {
      return ['#markup' => ''];
    }

    // Use language manager to get a human label when possible.
    $language_manager = \Drupal::languageManager();
    $language = $language_manager->getLanguage($langcode);
    $label = $language ? $language->getName() : $langcode;

    return ['#markup' => $label];
  }

  /**
   * Define an option to allow returning code instead of human name if needed.
   */
  public function defineOptions() {
    $options = parent::defineOptions();
    $options['display'] = ['default' => 'name'];
    return $options;
  }

}
