<?php

namespace Drupal\external_entities\Hook;

use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Hook implementations for External Entities.
 */
class ExternalEntityHooks {

  use StringTranslationTrait;

  /**
   * Implements hook_token_info().
   */
  #[Hook('token_info')]
  public function tokenInfo(): array {
    // REST client tokens.
    $output['types'] = [
      'xntt_sc_create' => [
        'name' => $this->t('External entity storage client entity creation tokens'),
        'description' => $this->t('Tokens for external entities storage client entity creation.'),
        'needs-data' => 'xntt_sc_create',
      ],
      'xntt_sc_read' => [
        'name' => $this->t('External entity storage client entity loading tokens'),
        'description' => $this->t('Tokens for external entities storage client entity loading.'),
        'needs-data' => 'xntt_sc_read',
      ],
      'xntt_sc_update' => [
        'name' => $this->t('External entity storage client entity update tokens'),
        'description' => $this->t('Tokens for external entities storage client entity update.'),
        'needs-data' => 'xntt_sc_update',
      ],
      'xntt_sc_delete' => [
        'name' => $this->t('External entity storage client entity removal tokens'),
        'description' => $this->t('Tokens for external entities storage client entity removal.'),
        'needs-data' => 'xntt_sc_delete',
      ],
      'xntt_sc_list' => [
        'name' => $this->t('External entity storage client entity list tokens'),
        'description' => $this->t('Tokens for external entities storage client entity listing.'),
        'needs-data' => 'xntt_sc_list',
      ],
      'xntt_sc_count' => [
        'name' => $this->t('External entity storage client entityt count tokens'),
        'description' => $this->t('Tokens for external entities storage client entity count.'),
        'needs-data' => 'xntt_sc_count',
      ],
    ];

    $output['tokens'] = [
      'xntt_sc_create' => [
        'id' => [
          'name' => $this->t('Storage client entity identifier (optional)'),
          'description' => $this->t('Entity identifier to use on the storage client side.'),
        ],
        'data' => [
          'name' => $this->t('Storage client entity raw data'),
          'description' => $this->t('Entity raw data structure.'),
        ],
      ],
      'xntt_sc_read' => [
        'id' => [
          'name' => $this->t('Storage client entity identifier'),
          'description' => $this->t('Entity identifier used on the storage client side.'),
        ],
      ],
      'xntt_sc_update' => [
        'id' => [
          'name' => $this->t('Storage client entity identifier'),
          'description' => $this->t('Entity identifier used on the storage client side.'),
        ],
        'data' => [
          'name' => $this->t('Storage client entity updated raw data'),
          'description' => $this->t('Entity updated raw data structure.'),
        ],
        'original' => [
          'name' => $this->t('Storage client entity original raw data'),
          'description' => $this->t('Entity original raw data structure.'),
        ],
      ],
      'xntt_sc_delete' => [
        'id' => [
          'name' => $this->t('Storage client entity identifier'),
          'description' => $this->t('Entity identifier used on the storage client side.'),
        ],
      ],
      'xntt_sc_list' => [
        'filters' => [
          'name' => $this->t('Filter parameters'),
          'description' => $this->t('Array of filtering parameters. Each value is an array of one of the two following structure: 1) type condition: ["field": the source field name the parameter applies to, "value": the value of the parameter or NULL, "operator": the "source-supported" operator of how the parameter should be applied] or 2) type sub-condition: ["conjunction": either "or" or "and", "conditions": an array of array of type condition described above or type sub-condition].'),
        ],
        'sorts' => [
          'name' => $this->t('Sorting parameters'),
          'description' => $this->t('Array of sorts, each value is an array with the following key-value pairs: ["field": the source field to sort by, "direction": the direction to sort on, "langcode": optional language code].'),
        ],
        'start' => [
          'name' => $this->t('Start'),
          'description' => $this->t('The index of the first item to return.'),
        ],
        'length' => [
          'name' => $this->t('Length'),
          'description' => $this->t('The number of items to return.'),
        ],
        'page' => [
          'name' => $this->t('Page number'),
          'description' => $this->t('The Drupal-side page number.'),
        ],
        'paging' => [
          'name' => $this->t('The storage client paging structure'),
          'description' => $this->t('Since storage client may use a different paging system, this structure contains client-specific paging parameters.'),
        ],
      ],
      'xntt_sc_count' => [
        'filters' => [
          'name' => $this->t('Filter parameters'),
          'description' => $this->t('Array of filtering parameters. Each value is an array of one of the two following structure: 1) type condition: ["field": the source field name the parameter applies to, "value": the value of the parameter or NULL, "operator": the "source-supported" operator of how the parameter should be applied] or 2) type sub-condition: ["conjunction": either "or" or "and", "conditions": an array of array of type condition described above or type sub-condition].'),
        ],
      ],
    ];

    return $output;
  }

  /**
   * Implements hook_tokens().
   */
  #[Hook('tokens')]
  public function tokens(
    $type,
    $tokens,
    array $data,
    array $options,
    BubbleableMetadata $bubbleable_metadata,
  ): array {
    $replacements = [];
    $token_data = [];
    $token_type_list = [
      'xntt_sc_create',
      'xntt_sc_read',
      'xntt_sc_update',
      'xntt_sc_delete',
      'xntt_sc_list',
      'xntt_sc_count',
    ];
    if (in_array($type, $token_type_list) && !empty($data[$type])) {
      $token_data = $data[$type] ?? [];
      if (!empty($token_data) || $options['clear']) {
        foreach ($tokens as $name => $original) {
          $parents = explode(':', $name);
          $replacements[$original] = NestedArray::getValue($token_data, $parents) ?? '';
        }
      }
    }
    return $replacements;
  }

}
