<?php

namespace Drupal\Tests\external_entities\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\external_entities\Plugin\ExternalEntities\DataAggregator\GroupAggregator;

/**
 * The class to test data aggregator.
 *
 * @group ExternalEntities
 */
class DataAggregatorUnitTest extends UnitTestCase {

  /**
   * Data provider for testMergeArray().
   *
   * Structure:
   * - expected_array
   * - array a
   * - array b
   * - merge_subarrays
   * - only_empty
   * - preserve_integer_keys
   * - no_new_keys
   * - test name.
   */
  public static function provideTestMergeArray() {
    return [
      // Full override.
      [
        ['id' => 'b'],
        ['id' => 'a'],
        ['id' => 'b'],
        FALSE,
        FALSE,
        FALSE,
        FALSE,
        'Merge full override 1',
      ],
      [
        ['id' => 'b', 'other_a' => 1],
        ['id' => 'a', 'other_a' => 1],
        ['id' => 'b'],
        FALSE,
        FALSE,
        FALSE,
        FALSE,
        'Merge full override 2',
      ],
      [
        ['id' => 'b', 'other_b' => 2],
        ['id' => 'a'],
        ['id' => 'b', 'other_b' => 2],
        FALSE,
        FALSE,
        FALSE,
        FALSE,
        'Merge full override 3',
      ],
      [
        ['id' => 'b', 'other_a' => 1, 'other_b' => 2],
        ['id' => 'a', 'other_a' => 1],
        ['id' => 'b', 'other_b' => 2],
        FALSE,
        FALSE,
        FALSE,
        FALSE,
        'Merge full override 4',
      ],
      [
        ['id' => 'b', 'field1' => ['d']],
        ['id' => 'a', 'field1' => 'c'],
        ['id' => 'b', 'field1' => ['d']],
        FALSE,
        FALSE,
        FALSE,
        FALSE,
        'Merge full override 5',
      ],
      [
        ['id' => 'b', 'field1' => 'd'],
        ['id' => 'a', 'field1' => ['c']],
        ['id' => 'b', 'field1' => 'd'],
        FALSE,
        FALSE,
        FALSE,
        FALSE,
        'Merge full override 6',
      ],
      [
        ['id' => 'b', 'field1' => ['d']],
        ['id' => 'a', 'field1' => ['c']],
        ['id' => 'b', 'field1' => ['d']],
        FALSE,
        FALSE,
        FALSE,
        FALSE,
        'Merge full override 7',
      ],
      [
        ['id' => 'b', 'field1' => 42, 'field2' => [66]],
        ['id' => 'a', 'field1' => NULL, 'field2' => []],
        ['id' => 'b', 'field1' => 42, 'field2' => [66]],
        FALSE,
        FALSE,
        FALSE,
        FALSE,
        'Merge full override 8',
      ],
      [
        ['id' => 'b', 'field1' => NULL, 'field2' => []],
        ['id' => 'a', 'field1' => 42, 'field2' => [66]],
        ['id' => 'b', 'field1' => NULL, 'field2' => []],
        FALSE,
        FALSE,
        FALSE,
        FALSE,
        'Merge full override 9',
      ],
      [
        ['id' => 'b', 'field1' => NULL],
        ['id' => 'a', 'field1' => ''],
        ['id' => 'b', 'field1' => NULL],
        FALSE,
        FALSE,
        FALSE,
        FALSE,
        'Merge full override 10',
      ],
      [
        ['id' => 'b', 'field1' => ['A' => 'beta']],
        ['id' => 'a', 'field1' => ['A' => 'alpha']],
        ['id' => 'b', 'field1' => ['A' => 'beta']],
        FALSE,
        FALSE,
        FALSE,
        FALSE,
        'Merge full override 11',
      ],
      [
        ['id' => 'b', 'field1' => ['B' => 'beta']],
        ['id' => 'a', 'field1' => ['A' => 'alpha']],
        ['id' => 'b', 'field1' => ['B' => 'beta']],
        FALSE,
        FALSE,
        FALSE,
        FALSE,
        'Merge full override 12',
      ],
      [
        ['id' => 'b', 'field1' => ['A' => 'alpha', 'B' => 'beta', 'C' => 'gamma']],
        ['id' => 'a', 'field1' => ['A' => 'abc', 'C' => '', 'D' => 'delta']],
        ['id' => 'b', 'field1' => ['A' => 'alpha', 'B' => 'beta', 'C' => 'gamma']],
        FALSE,
        FALSE,
        FALSE,
        FALSE,
        'Merge full override 13',
      ],
      [
        ['id' => 'b', 'field1' => ['c', 'd']],
        ['id' => 'a', 'field1' => ['a', 'b']],
        ['id' => 'b', 'field1' => ['c', 'd']],
        FALSE,
        FALSE,
        FALSE,
        FALSE,
        'Merge full override 14',
      ],
      [
        ['id' => 'b', 'field1' => [2 => 'c', 4 => 'd']],
        ['id' => 'a', 'field1' => [2 => 'a', 5 => 'b']],
        ['id' => 'b', 'field1' => [2 => 'c', 4 => 'd']],
        FALSE,
        FALSE,
        FALSE,
        FALSE,
        'Merge full override 15',
      ],
      [
        ['id' => 'b', 'field1' => ['a' => ['b' => 806, 'd' => 42]]],
        ['id' => 'a', 'field1' => ['a' => ['b' => 33, 'c' => 7]]],
        ['id' => 'b', 'field1' => ['a' => ['b' => 806, 'd' => 42]]],
        FALSE,
        FALSE,
        FALSE,
        FALSE,
        'Merge full override 16',
      ],
      // Deep merge.
      [
        ['id' => 'b'],
        ['id' => 'a'],
        ['id' => 'b'],
        TRUE,
        FALSE,
        FALSE,
        FALSE,
        'Merge deep 1',
      ],
      [
        ['id' => 'b', 'other_a' => 1],
        ['id' => 'a', 'other_a' => 1],
        ['id' => 'b'],
        TRUE,
        FALSE,
        FALSE,
        FALSE,
        'Merge deep 2',
      ],
      [
        ['id' => 'b', 'other_b' => 2],
        ['id' => 'a'],
        ['id' => 'b', 'other_b' => 2],
        TRUE,
        FALSE,
        FALSE,
        FALSE,
        'Merge deep 3',
      ],
      [
        ['id' => 'b', 'other_a' => 1, 'other_b' => 2],
        ['id' => 'a', 'other_a' => 1],
        ['id' => 'b', 'other_b' => 2],
        TRUE,
        FALSE,
        FALSE,
        FALSE,
        'Merge deep 4',
      ],
      [
        ['id' => 'b', 'field1' => ['c', 'd']],
        ['id' => 'a', 'field1' => 'c'],
        ['id' => 'b', 'field1' => ['d']],
        TRUE,
        FALSE,
        FALSE,
        FALSE,
        'Merge deep 5',
      ],
      [
        ['id' => 'b', 'field1' => ['c', 'd']],
        ['id' => 'a', 'field1' => ['c']],
        ['id' => 'b', 'field1' => 'd'],
        TRUE,
        FALSE,
        FALSE,
        FALSE,
        'Merge deep 6',
      ],
      [
        ['id' => 'b', 'field1' => ['c', 'd']],
        ['id' => 'a', 'field1' => ['c']],
        ['id' => 'b', 'field1' => ['d']],
        TRUE,
        FALSE,
        FALSE,
        FALSE,
        'Merge deep 7',
      ],
      [
        ['id' => 'b', 'field1' => 42, 'field2' => [66]],
        ['id' => 'a', 'field1' => NULL, 'field2' => []],
        ['id' => 'b', 'field1' => 42, 'field2' => [66]],
        TRUE,
        FALSE,
        FALSE,
        FALSE,
        'Merge deep 8',
      ],
      [
        ['id' => 'b', 'field1' => NULL, 'field2' => [66]],
        ['id' => 'a', 'field1' => 42, 'field2' => [66]],
        ['id' => 'b', 'field1' => NULL, 'field2' => []],
        TRUE,
        FALSE,
        FALSE,
        FALSE,
        'Merge deep 9',
      ],
      [
        ['id' => 'b', 'field1' => NULL],
        ['id' => 'a', 'field1' => ''],
        ['id' => 'b', 'field1' => NULL],
        TRUE,
        FALSE,
        FALSE,
        FALSE,
        'Merge deep 10',
      ],
      [
        ['id' => 'b', 'field1' => ['A' => 'beta']],
        ['id' => 'a', 'field1' => ['A' => 'alpha']],
        ['id' => 'b', 'field1' => ['A' => 'beta']],
        TRUE,
        FALSE,
        FALSE,
        FALSE,
        'Merge deep 11',
      ],
      [
        ['id' => 'b', 'field1' => ['A' => 'alpha', 'B' => 'beta']],
        ['id' => 'a', 'field1' => ['A' => 'alpha']],
        ['id' => 'b', 'field1' => ['B' => 'beta']],
        TRUE,
        FALSE,
        FALSE,
        FALSE,
        'Merge deep 12',
      ],
      [
        ['id' => 'b', 'field1' => ['A' => 'alpha', 'B' => 'beta', 'C' => 'gamma', 'D' => 'delta']],
        ['id' => 'a', 'field1' => ['A' => 'abc', 'C' => '', 'D' => 'delta']],
        ['id' => 'b', 'field1' => ['A' => 'alpha', 'B' => 'beta', 'C' => 'gamma']],
        TRUE,
        FALSE,
        FALSE,
        FALSE,
        'Merge deep 13',
      ],
      [
        ['id' => 'b', 'field1' => ['a', 'b', 'c', 'd']],
        ['id' => 'a', 'field1' => ['a', 'b']],
        ['id' => 'b', 'field1' => ['c', 'd']],
        TRUE,
        FALSE,
        FALSE,
        FALSE,
        'Merge deep 14',
      ],
      [
        ['id' => 'b', 'field1' => ['a', 'b', 'c', 'd']],
        ['id' => 'a', 'field1' => [2 => 'a', 5 => 'b']],
        ['id' => 'b', 'field1' => [2 => 'c', 4 => 'd']],
        TRUE,
        FALSE,
        FALSE,
        FALSE,
        'Merge deep 15',
      ],
      [
        ['id' => 'b', 'field1' => ['a' => ['b' => 806, 'c' => 7, 'd' => 42]]],
        ['id' => 'a', 'field1' => ['a' => ['b' => 33, 'c' => 7]]],
        ['id' => 'b', 'field1' => ['a' => ['b' => 806, 'd' => 42]]],
        TRUE,
        FALSE,
        FALSE,
        FALSE,
        'Merge deep 16',
      ],
      // Override empty.
      [
        ['id' => 'a'],
        ['id' => 'a'],
        ['id' => 'b'],
        FALSE,
        TRUE,
        FALSE,
        FALSE,
        'Merge override empty 1',
      ],
      [
        ['id' => 'a', 'other_a' => 1],
        ['id' => 'a', 'other_a' => 1],
        ['id' => 'b'],
        FALSE,
        TRUE,
        FALSE,
        FALSE,
        'Merge override empty 2',
      ],
      [
        ['id' => 'a', 'other_b' => 2],
        ['id' => 'a'],
        ['id' => 'b', 'other_b' => 2],
        FALSE,
        TRUE,
        FALSE,
        FALSE,
        'Merge override empty 3',
      ],
      [
        ['id' => 'a', 'other_a' => 1, 'other_b' => 2],
        ['id' => 'a', 'other_a' => 1],
        ['id' => 'b', 'other_b' => 2],
        FALSE,
        TRUE,
        FALSE,
        FALSE,
        'Merge override empty 4',
      ],
      [
        ['id' => 'a', 'field1' => 'c'],
        ['id' => 'a', 'field1' => 'c'],
        ['id' => 'b', 'field1' => ['d']],
        FALSE,
        TRUE,
        FALSE,
        FALSE,
        'Merge override empty 5',
      ],
      [
        ['id' => 'a', 'field1' => ['c']],
        ['id' => 'a', 'field1' => ['c']],
        ['id' => 'b', 'field1' => 'd'],
        FALSE,
        TRUE,
        FALSE,
        FALSE,
        'Merge override empty 6',
      ],
      [
        ['id' => 'a', 'field1' => ['c']],
        ['id' => 'a', 'field1' => ['c']],
        ['id' => 'b', 'field1' => ['d']],
        FALSE,
        TRUE,
        FALSE,
        FALSE,
        'Merge override empty 7',
      ],
      [
        ['id' => 'a', 'field1' => 42, 'field2' => [66]],
        ['id' => 'a', 'field1' => NULL, 'field2' => []],
        ['id' => 'b', 'field1' => 42, 'field2' => [66]],
        FALSE,
        TRUE,
        FALSE,
        FALSE,
        'Merge override empty 8',
      ],
      [
        ['id' => 'a', 'field1' => 42, 'field2' => [66]],
        ['id' => 'a', 'field1' => 42, 'field2' => [66]],
        ['id' => 'b', 'field1' => NULL, 'field2' => []],
        FALSE,
        TRUE,
        FALSE,
        FALSE,
        'Merge override empty 9',
      ],
      [
        ['id' => 'a', 'field1' => ''],
        ['id' => 'a', 'field1' => ''],
        ['id' => 'b', 'field1' => NULL],
        FALSE,
        TRUE,
        FALSE,
        FALSE,
        'Merge override empty 10',
      ],
      [
        ['id' => 'a', 'field1' => ['A' => 'alpha']],
        ['id' => 'a', 'field1' => ['A' => 'alpha']],
        ['id' => 'b', 'field1' => ['A' => 'beta']],
        FALSE,
        TRUE,
        FALSE,
        FALSE,
        'Merge override empty 11',
      ],
      [
        ['id' => 'a', 'field1' => ['A' => 'alpha']],
        ['id' => 'a', 'field1' => ['A' => 'alpha']],
        ['id' => 'b', 'field1' => ['B' => 'beta']],
        FALSE,
        TRUE,
        FALSE,
        FALSE,
        'Merge override empty 12',
      ],
      [
        ['id' => 'a', 'field1' => ['A' => 'abc', 'C' => '', 'D' => 'delta']],
        ['id' => 'a', 'field1' => ['A' => 'abc', 'C' => '', 'D' => 'delta']],
        ['id' => 'b', 'field1' => ['A' => 'alpha', 'B' => 'beta', 'C' => 'gamma']],
        FALSE,
        TRUE,
        FALSE,
        FALSE,
        'Merge override empty 13',
      ],
      [
        ['id' => 'a', 'field1' => ['a', 'b']],
        ['id' => 'a', 'field1' => ['a', 'b']],
        ['id' => 'b', 'field1' => ['c', 'd']],
        FALSE,
        TRUE,
        FALSE,
        FALSE,
        'Merge override empty 14',
      ],
      [
        ['id' => 'a', 'field1' => [2 => 'a', 5 => 'b']],
        ['id' => 'a', 'field1' => [2 => 'a', 5 => 'b']],
        ['id' => 'b', 'field1' => [2 => 'c', 4 => 'd']],
        FALSE,
        TRUE,
        FALSE,
        FALSE,
        'Merge override empty 15',
      ],
      [
        ['id' => 'a', 'field1' => ['a' => ['b' => 33, 'c' => 7]]],
        ['id' => 'a', 'field1' => ['a' => ['b' => 33, 'c' => 7]]],
        ['id' => 'b', 'field1' => ['a' => ['b' => 806, 'd' => 42]]],
        FALSE,
        TRUE,
        FALSE,
        FALSE,
        'Merge override empty 16',
      ],
      // Deep merge empty.
      [
        ['id' => 'a'],
        ['id' => 'a'],
        ['id' => 'b'],
        TRUE,
        TRUE,
        FALSE,
        FALSE,
        'Merge deep empty 1',
      ],
      [
        ['id' => 'a', 'other_a' => 1],
        ['id' => 'a', 'other_a' => 1],
        ['id' => 'b'],
        TRUE,
        TRUE,
        FALSE,
        FALSE,
        'Merge deep empty 2',
      ],
      [
        ['id' => 'a', 'other_b' => 2],
        ['id' => 'a'],
        ['id' => 'b', 'other_b' => 2],
        TRUE,
        TRUE,
        FALSE,
        FALSE,
        'Merge deep empty 3',
      ],
      [
        ['id' => 'a', 'other_a' => 1, 'other_b' => 2],
        ['id' => 'a', 'other_a' => 1],
        ['id' => 'b', 'other_b' => 2],
        TRUE,
        TRUE,
        FALSE,
        FALSE,
        'Merge deep empty 4',
      ],
      [
        ['id' => 'a', 'field1' => ['c', 'd']],
        ['id' => 'a', 'field1' => 'c'],
        ['id' => 'b', 'field1' => ['d']],
        TRUE,
        TRUE,
        FALSE,
        FALSE,
        'Merge deep empty 5',
      ],
      [
        ['id' => 'a', 'field1' => ['c', 'd']],
        ['id' => 'a', 'field1' => ['c']],
        ['id' => 'b', 'field1' => 'd'],
        TRUE,
        TRUE,
        FALSE,
        FALSE,
        'Merge deep empty 6',
      ],
      [
        ['id' => 'a', 'field1' => ['c', 'd']],
        ['id' => 'a', 'field1' => ['c']],
        ['id' => 'b', 'field1' => ['d']],
        TRUE,
        TRUE,
        FALSE,
        FALSE,
        'Merge deep empty 7',
      ],
      [
        ['id' => 'a', 'field1' => 42, 'field2' => [66]],
        ['id' => 'a', 'field1' => NULL, 'field2' => []],
        ['id' => 'b', 'field1' => 42, 'field2' => [66]],
        TRUE,
        TRUE,
        FALSE,
        FALSE,
        'Merge deep empty 8',
      ],
      [
        ['id' => 'a', 'field1' => 42, 'field2' => [66]],
        ['id' => 'a', 'field1' => 42, 'field2' => [66]],
        ['id' => 'b', 'field1' => NULL, 'field2' => []],
        TRUE,
        TRUE,
        FALSE,
        FALSE,
        'Merge deep empty 9',
      ],
      [
        ['id' => 'a', 'field1' => ''],
        ['id' => 'a', 'field1' => ''],
        ['id' => 'b', 'field1' => NULL],
        TRUE,
        TRUE,
        FALSE,
        FALSE,
        'Merge deep empty 10',
      ],
      [
        ['id' => 'a', 'field1' => ['A' => 'alpha']],
        ['id' => 'a', 'field1' => ['A' => 'alpha']],
        ['id' => 'b', 'field1' => ['A' => 'beta']],
        TRUE,
        TRUE,
        FALSE,
        FALSE,
        'Merge deep empty 11',
      ],
      [
        ['id' => 'a', 'field1' => ['A' => 'alpha', 'B' => 'beta']],
        ['id' => 'a', 'field1' => ['A' => 'alpha']],
        ['id' => 'b', 'field1' => ['B' => 'beta']],
        TRUE,
        TRUE,
        FALSE,
        FALSE,
        'Merge deep empty 12',
      ],
      [
        ['id' => 'a', 'field1' => ['A' => 'abc', 'B' => 'beta', 'C' => 'gamma', 'D' => 'delta']],
        ['id' => 'a', 'field1' => ['A' => 'abc', 'C' => '', 'D' => 'delta']],
        ['id' => 'b', 'field1' => ['A' => 'alpha', 'B' => 'beta', 'C' => 'gamma']],
        TRUE,
        TRUE,
        FALSE,
        FALSE,
        'Merge deep empty 13',
      ],
      [
        ['id' => 'a', 'field1' => ['a', 'b', 'c', 'd']],
        ['id' => 'a', 'field1' => ['a', 'b']],
        ['id' => 'b', 'field1' => ['c', 'd']],
        TRUE,
        TRUE,
        FALSE,
        FALSE,
        'Merge deep empty 14',
      ],
      [
        ['id' => 'a', 'field1' => ['a', 'b', 'c', 'd']],
        ['id' => 'a', 'field1' => [2 => 'a', 5 => 'b']],
        ['id' => 'b', 'field1' => [2 => 'c', 4 => 'd']],
        TRUE,
        TRUE,
        FALSE,
        FALSE,
        'Merge deep empty 15',
      ],
      [
        ['id' => 'a', 'field1' => ['a' => ['b' => 33, 'c' => 7, 'd' => 42]]],
        ['id' => 'a', 'field1' => ['a' => ['b' => 33, 'c' => 7]]],
        ['id' => 'b', 'field1' => ['a' => ['b' => 806, 'd' => 42]]],
        TRUE,
        TRUE,
        FALSE,
        FALSE,
        'Merge deep empty 16',
      ],
      // Preserve int keys.
      [
        ['id' => 'b'],
        ['id' => 'a'],
        ['id' => 'b'],
        FALSE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys 1',
      ],
      [
        ['b'],
        ['a'],
        ['b'],
        FALSE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys 2',
      ],
      [
        ['b', 'a'],
        [1 => 'a'],
        ['b'],
        FALSE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys 3',
      ],
      [
        ['b', 'c', 'd'],
        ['a', 'c'],
        ['b', 2 => 'd'],
        FALSE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys 4',
      ],
      [
        ['id' => 'b', 'field1' => ['d']],
        ['id' => 'a', 'field1' => 'c'],
        ['id' => 'b', 'field1' => ['d']],
        FALSE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys 5',
      ],
      [
        ['id' => 'b', 'field1' => 'd'],
        ['id' => 'a', 'field1' => ['c']],
        ['id' => 'b', 'field1' => 'd'],
        FALSE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys 6',
      ],
      [
        ['id' => 'b', 'field1' => ['d']],
        ['id' => 'a', 'field1' => ['c']],
        ['id' => 'b', 'field1' => ['d']],
        FALSE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys 7',
      ],
      [
        ['id' => 'b', 'field1' => 42, 'field2' => [66]],
        ['id' => 'a', 'field1' => NULL, 'field2' => []],
        ['id' => 'b', 'field1' => 42, 'field2' => [66]],
        FALSE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys 8',
      ],
      [
        ['id' => 'b', 'field1' => NULL, 'field2' => []],
        ['id' => 'a', 'field1' => 42, 'field2' => [66]],
        ['id' => 'b', 'field1' => NULL, 'field2' => []],
        FALSE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys 9',
      ],
      [
        ['id' => 'b', 'field1' => NULL],
        ['id' => 'a', 'field1' => ''],
        ['id' => 'b', 'field1' => NULL],
        FALSE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys 10',
      ],
      [
        ['id' => 'b', 'field1' => ['A' => 'beta']],
        ['id' => 'a', 'field1' => ['A' => 'alpha']],
        ['id' => 'b', 'field1' => ['A' => 'beta']],
        FALSE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys 11',
      ],
      [
        ['id' => 'b', 'field1' => ['B' => 'beta']],
        ['id' => 'a', 'field1' => ['A' => 'alpha']],
        ['id' => 'b', 'field1' => ['B' => 'beta']],
        FALSE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys 12',
      ],
      [
        ['id' => 'b', 'field1' => ['A' => 'alpha', 'B' => 'beta', 'C' => 'gamma']],
        ['id' => 'a', 'field1' => ['A' => 'abc', 'C' => '', 'D' => 'delta']],
        ['id' => 'b', 'field1' => ['A' => 'alpha', 'B' => 'beta', 'C' => 'gamma']],
        FALSE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys 13',
      ],
      [
        ['id' => 'b', 'field1' => ['c', 'd']],
        ['id' => 'a', 'field1' => ['a', 'b']],
        ['id' => 'b', 'field1' => ['c', 'd']],
        FALSE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys 14',
      ],
      [
        ['id' => 'b', 'field1' => [2 => 'c', 4 => 'd']],
        ['id' => 'a', 'field1' => [2 => 'a', 5 => 'b']],
        ['id' => 'b', 'field1' => [2 => 'c', 4 => 'd']],
        FALSE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys 15',
      ],
      [
        ['id' => 'b', 'field1' => ['a' => ['b' => 806, 'd' => 42]]],
        ['id' => 'a', 'field1' => ['a' => ['b' => 33, 'c' => 7]]],
        ['id' => 'b', 'field1' => ['a' => ['b' => 806, 'd' => 42]]],
        FALSE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys 16',
      ],
      // Preserve int keys deep.
      [
        ['id' => 'b'],
        ['id' => 'a'],
        ['id' => 'b'],
        TRUE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys deep 1',
      ],
      [
        ['b'],
        ['a'],
        ['b'],
        TRUE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys deep 2',
      ],
      [
        ['b', 'a'],
        [1 => 'a'],
        ['b'],
        TRUE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys deep 3',
      ],
      [
        ['b', 'c', 'd'],
        ['a', 'c'],
        ['b', 2 => 'd'],
        TRUE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys deep 4',
      ],
      [
        ['id' => 'b', 'field1' => ['d']],
        ['id' => 'a', 'field1' => 'c'],
        ['id' => 'b', 'field1' => ['d']],
        TRUE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys deep 5',
      ],
      [
        ['id' => 'b', 'field1' => ['d']],
        ['id' => 'a', 'field1' => ['c']],
        ['id' => 'b', 'field1' => 'd'],
        TRUE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys deep 6',
      ],
      [
        ['id' => 'b', 'field1' => ['d']],
        ['id' => 'a', 'field1' => ['c']],
        ['id' => 'b', 'field1' => ['d']],
        TRUE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys deep 7',
      ],
      [
        ['id' => 'b', 'field1' => 42, 'field2' => [66]],
        ['id' => 'a', 'field1' => NULL, 'field2' => []],
        ['id' => 'b', 'field1' => 42, 'field2' => [66]],
        TRUE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys deep 8',
      ],
      [
        ['id' => 'b', 'field1' => NULL, 'field2' => [66]],
        ['id' => 'a', 'field1' => 42, 'field2' => [66]],
        ['id' => 'b', 'field1' => NULL, 'field2' => []],
        TRUE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys deep 9',
      ],
      [
        ['id' => 'b', 'field1' => NULL],
        ['id' => 'a', 'field1' => ''],
        ['id' => 'b', 'field1' => NULL],
        TRUE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys deep 10',
      ],
      [
        ['id' => 'b', 'field1' => ['A' => 'beta']],
        ['id' => 'a', 'field1' => ['A' => 'alpha']],
        ['id' => 'b', 'field1' => ['A' => 'beta']],
        TRUE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys deep 11',
      ],
      [
        ['id' => 'b', 'field1' => ['A' => 'alpha', 'B' => 'beta']],
        ['id' => 'a', 'field1' => ['A' => 'alpha']],
        ['id' => 'b', 'field1' => ['B' => 'beta']],
        TRUE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys deep 12',
      ],
      [
        ['id' => 'b', 'field1' => ['A' => 'alpha', 'B' => 'beta', 'C' => 'gamma', 'D' => 'delta']],
        ['id' => 'a', 'field1' => ['A' => 'abc', 'C' => '', 'D' => 'delta']],
        ['id' => 'b', 'field1' => ['A' => 'alpha', 'B' => 'beta', 'C' => 'gamma']],
        TRUE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys deep 13',
      ],
      [
        ['id' => 'b', 'field1' => ['c', 'd']],
        ['id' => 'a', 'field1' => ['a', 'b']],
        ['id' => 'b', 'field1' => ['c', 'd']],
        TRUE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys deep 14',
      ],
      [
        ['id' => 'b', 'field1' => [2 => 'c', 4 => 'd', 5 => 'b']],
        ['id' => 'a', 'field1' => [2 => 'a', 5 => 'b']],
        ['id' => 'b', 'field1' => [2 => 'c', 4 => 'd']],
        TRUE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys deep 15',
      ],
      [
        ['id' => 'b', 'field1' => ['a' => ['b' => 806, 'c' => 7, 'd' => 42]]],
        ['id' => 'a', 'field1' => ['a' => ['b' => 33, 'c' => 7]]],
        ['id' => 'b', 'field1' => ['a' => ['b' => 806, 'd' => 42]]],
        TRUE,
        FALSE,
        TRUE,
        FALSE,
        'Merge preserve int keys deep 16',
      ],
      // Deep merge no new keys.
      [
        ['id' => 'b'],
        ['id' => 'a'],
        ['id' => 'b'],
        TRUE,
        FALSE,
        FALSE,
        TRUE,
        'Merge deep no new keys 1',
      ],
      [
        ['id' => 'b', 'other_a' => 1],
        ['id' => 'a', 'other_a' => 1],
        ['id' => 'b'],
        TRUE,
        FALSE,
        FALSE,
        TRUE,
        'Merge deep no new keys 2',
      ],
      [
        ['id' => 'b'],
        ['id' => 'a'],
        ['id' => 'b', 'other_b' => 2],
        TRUE,
        FALSE,
        FALSE,
        TRUE,
        'Merge deep no new keys 3',
      ],
      [
        ['id' => 'b', 'other_a' => 1],
        ['id' => 'a', 'other_a' => 1],
        ['id' => 'b', 'other_b' => 2],
        TRUE,
        FALSE,
        FALSE,
        TRUE,
        'Merge deep no new keys 4',
      ],
      [
        ['id' => 'b', 'field1' => ['d']],
        ['id' => 'a', 'field1' => 'c'],
        ['id' => 'b', 'field1' => ['d']],
        TRUE,
        FALSE,
        FALSE,
        TRUE,
        'Merge deep no new keys 5',
      ],
      [
        ['id' => 'b', 'field1' => ['d']],
        ['id' => 'a', 'field1' => ['c']],
        ['id' => 'b', 'field1' => 'd'],
        TRUE,
        FALSE,
        FALSE,
        TRUE,
        'Merge deep no new keys 6',
      ],
      [
        ['id' => 'b', 'field1' => ['d']],
        ['id' => 'a', 'field1' => ['c']],
        ['id' => 'b', 'field1' => ['d']],
        TRUE,
        FALSE,
        FALSE,
        TRUE,
        'Merge deep no new keys 7',
      ],
      [
        ['id' => 'b', 'field1' => 42, 'field2' => []],
        ['id' => 'a', 'field1' => NULL, 'field2' => []],
        ['id' => 'b', 'field1' => 42, 'field2' => [66]],
        TRUE,
        FALSE,
        FALSE,
        TRUE,
        'Merge deep no new keys 8',
      ],
      [
        ['id' => 'b', 'field1' => NULL, 'field2' => [66]],
        ['id' => 'a', 'field1' => 42, 'field2' => [66]],
        ['id' => 'b', 'field1' => NULL, 'field2' => []],
        TRUE,
        FALSE,
        FALSE,
        TRUE,
        'Merge deep no new keys 9',
      ],
      [
        ['id' => 'b', 'field1' => NULL],
        ['id' => 'a', 'field1' => ''],
        ['id' => 'b', 'field1' => NULL],
        TRUE,
        FALSE,
        FALSE,
        TRUE,
        'Merge deep no new keys 10',
      ],
      [
        ['id' => 'b', 'field1' => ['A' => 'beta']],
        ['id' => 'a', 'field1' => ['A' => 'alpha']],
        ['id' => 'b', 'field1' => ['A' => 'beta']],
        TRUE,
        FALSE,
        FALSE,
        TRUE,
        'Merge deep no new keys 11',
      ],
      [
        ['id' => 'b', 'field1' => ['A' => 'alpha']],
        ['id' => 'a', 'field1' => ['A' => 'alpha']],
        ['id' => 'b', 'field1' => ['B' => 'beta']],
        TRUE,
        FALSE,
        FALSE,
        TRUE,
        'Merge deep no new keys 12',
      ],
      [
        ['id' => 'b', 'field1' => ['A' => 'alpha', 'C' => 'gamma', 'D' => 'delta']],
        ['id' => 'a', 'field1' => ['A' => 'abc', 'C' => '', 'D' => 'delta']],
        ['id' => 'b', 'field1' => ['A' => 'alpha', 'B' => 'beta', 'C' => 'gamma']],
        TRUE,
        FALSE,
        FALSE,
        TRUE,
        'Merge deep no new keys 13',
      ],
      [
        ['id' => 'b', 'field1' => ['c', 'd']],
        ['id' => 'a', 'field1' => ['a', 'b']],
        ['id' => 'b', 'field1' => ['c', 'd']],
        TRUE,
        FALSE,
        FALSE,
        TRUE,
        'Merge deep no new keys 14',
      ],
      [
        ['id' => 'b', 'field1' => [2 => 'c', 5 => 'b']],
        ['id' => 'a', 'field1' => [2 => 'a', 5 => 'b']],
        ['id' => 'b', 'field1' => [2 => 'c', 4 => 'd']],
        TRUE,
        FALSE,
        FALSE,
        TRUE,
        'Merge deep no new keys 15',
      ],
      [
        ['id' => 'b', 'field1' => ['a' => ['b' => 806, 'c' => 7]]],
        ['id' => 'a', 'field1' => ['a' => ['b' => 33, 'c' => 7]]],
        ['id' => 'b', 'field1' => ['a' => ['b' => 806, 'd' => 42]]],
        TRUE,
        FALSE,
        FALSE,
        TRUE,
        'Merge deep no new keys 16',
      ],
    ];
  }

  /**
   * Tests multi-storages.
   *
   * @dataProvider provideTestMergeArray
   */
  public function testMergeArray(
    array $expected_array,
    array $a,
    array $b,
    bool $merge_subarrays,
    bool $only_empty,
    bool $preserve_integer_keys,
    bool $no_new_keys,
    string $test_name,
  ) {
    $result = GroupAggregator::mergeArrays($a, $b, $merge_subarrays, $only_empty, $preserve_integer_keys, $no_new_keys);
    $this->assertEquals($expected_array, $result, $test_name);
  }

}
