<?php

namespace Drupal\external_link_a11y\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Utility\Error;
use Drupal\link\Plugin\Field\FieldFormatter\LinkFormatter;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'external_link_a11y' formatter.
 */
#[FieldFormatter(
  id: 'external_link_a11y',
  label: new TranslatableMarkup('Link (target blank when external a11y)'),
  field_types: [
    'link',
  ],
)]
class ExternalLinkA11yFormatter extends LinkFormatter {

  /**
   * The logger factory service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->loggerFactory = $container->get('logger.factory');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'rel_noopener' => FALSE,
      'rel_noreferrer' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {

    $elements = parent::settingsForm($form, $form_state);
    unset($elements['target']);

    $elements['rel_noopener'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Add rel="noopener" to external links'),
      '#return_value' => 'nofollow',
      '#default_value' => $this->getSetting('rel_noopener') ?? FALSE,
    ];

    $elements['rel_noreferrer'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Add rel="noreferrer" to external links'),
      '#return_value' => 'nofollow',
      '#default_value' => $this->getSetting('rel_noreferrer') ?? FALSE,
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();

    $settings = $this->getSettings();

    if ($settings['rel_noopener'] || $settings['rel_noreferrer']) {
      foreach (['noopener', 'noreferrer'] as $rel) {
        if (!empty($settings['rel_' . $rel])) {
          $summary[] = $this->t('Add rel="@rel"', ['@rel' => $rel]);
        }
      }
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    try {
      $elements = parent::viewElements($items, $langcode);
      $settings = $this->getSettings();

      foreach ($elements as &$element) {
        /** @var \Drupal\Core\Url $url */
        $url = &$element['#url'];
        $attributes = $url->getOption('attributes') ?? [];
        if ($url->isExternal() && !isset($attributes['target'])) {
          $attributes['target'] = '_blank';
        }
        // Adds info that link will be opened in a new window if target blank.
        if (isset($attributes['target']) && $attributes['target'] === '_blank') {
          $params = ['@title' => $element['#title']];
          $attributes['title'] = $this->t('@title (Open in new window)', $params);
          $element['#title'] = $this->t('@title<span class="visually-hidden sr-only"> (Open in new window)</span>', $params);
          if ($settings['rel_noopener'] || $settings['rel_noreferrer']) {
            $rel = trim($attributes['rel'] ?? '');
            $rel_values = $rel === '' ? [] : explode(' ', $rel);
            foreach (['noopener', 'noreferrer'] as $rel_to_add) {
              if ($settings['rel_' . $rel_to_add]) {
                if (!in_array($rel_to_add, $rel_values)) {
                  $rel_values[] = $rel_to_add;
                }
              }
            }
            if (!empty($rel_values)) {
              $attributes['rel'] = implode(' ', $rel_values);
            }
          }
        }
        if (!empty($attributes)) {
          $url->setOption('attributes', $attributes);
        }
      }

      return $elements;
    }
    catch (\Exception $e) {
      Error::logException($this->loggerFactory->get('external_link_a11y'), $e);
      return [];
    }
  }

}
