<?php

namespace Drupal\external_link_a11y\Plugin\Filter;

use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\filter\FilterProcessResult;
use Drupal\filter\Plugin\FilterBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A filter that improves accessibility of external links.
 *
 * @Filter(
 *   id = "external_link_a11y",
 *   title = @Translation("Accessible external link filter"),
 *   description = @Translation("Improves accessibility of external links in WYSIWYG fields."),
 *   type = Drupal\filter\Plugin\FilterInterface::TYPE_TRANSFORM_REVERSIBLE,
 *   settings = {
 *     "external_url" = true,
 *     "target_blank" = true,
 *     "a11y_new_window" = true,
 *     "rel_noopener" = false,
 *     "rel_noreferrer" = false,
 *     "css_classes" = "",
 *     "html_suffix" = "",
 *   },
 *   weight = 0
 * )
 */
class ExternalLinkA11yFilter extends FilterBase implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public function process($text, $langcode) {
    $result = new FilterProcessResult($text);

    $dom = Html::load($text);
    $anchors = $dom->getElementsByTagName('a');

    if ($anchors->length > 0) {
      $updated = FALSE;
      foreach ($anchors as $anchor) {

        $is_opening_in_new_window = FALSE;

        if ($this->settings['external_url']) {
          $is_opening_in_new_window = UrlHelper::isExternal($anchor->getAttribute('href'));
        }
        if (!$is_opening_in_new_window && $this->settings['target_blank']) {
          $is_opening_in_new_window = $anchor->getAttribute('target') === '_blank';
        }

        if ($is_opening_in_new_window) {
          if ($this->settings['external_url'] && empty($anchor->getAttribute('target'))) {
            $anchor->setAttribute('target', '_blank');
            $updated = TRUE;
          }
          if ($this->settings['rel_noopener'] || $this->settings['rel_noreferrer']) {
            $rel = trim($anchor->getAttribute('rel'));
            $rel_values = $rel === '' ? [] : explode(' ', $rel);
            foreach (['noopener', 'noreferrer'] as $rel_to_add) {
              if ($this->settings['rel_' . $rel_to_add]) {
                if (!in_array($rel_to_add, $rel_values)) {
                  $rel_values[] = $rel_to_add;
                }
              }
            }
            if (!empty($rel_values)) {
              $anchor->setAttribute('rel', implode(' ', $rel_values));
            }
          }
          if ($this->settings['a11y_new_window']) {
            if (empty($anchor->getAttribute('title')) && !empty($anchor->textContent)) {
              $title_params = ['@title' => $anchor->textContent];
              $title_attribute = $this->t('@title (Open in new window)', $title_params);
              $title_attribute = html_entity_decode($title_attribute, ENT_QUOTES, 'UTF-8');
              $anchor->setAttribute('title', $title_attribute);
            }
            if (!empty($this->settings['css_classes'])) {
              $css_classes = explode(' ', $this->settings['css_classes']);
              $existing_classes = explode(' ', $anchor->getAttribute('class'));
              $new_classes = array_unique(array_merge($existing_classes, $css_classes));
              $anchor->setAttribute('class', trim(implode(' ', $new_classes)));
            }
            $span = $dom->createElement('span');
            $span->setAttribute('class', 'visually-hidden sr-only');
            $span->nodeValue = ' (' . $this->t('Open in new window') . ')';
            $anchor->appendChild($span);
            $updated = TRUE;
          }
          if (!empty($this->settings['html_suffix'])) {
            $fragment = $dom->createDocumentFragment();
            $fragment->appendXML($this->settings['html_suffix']);
            $anchor->appendChild($fragment);
            $updated = TRUE;
          }
        }
      }

      if ($updated) {
        $result->setProcessedText(Html::serialize($dom));
      }
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form['external_url'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Apply if external URL'),
      '#default_value' => $this->settings['external_url'],
      '#description' => $this->t("Filter will be applied if to links having an external URL in their href attribute."),
    ];
    $form['target_blank'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Apply if target is _blank'),
      '#default_value' => $this->settings['target_blank'],
      '#description' => $this->t("Filter will be applied to links having a target _blank attribute."),
    ];
    $form['a11y_new_window'] = [
      '#type' => 'checkbox',
      '#title' => $this->t("Add visually hidden 'Open in new window' message for accessibility"),
      '#default_value' => $this->settings['a11y_new_window'],
      '#description' => $this->t(
        "A visually hidden span element will be appended to the link element containing 'Open in new window' text."
      ),
    ];
    $form['rel_noopener'] = [
      '#type' => 'checkbox',
      '#title' => $this->t("Add rel attribute 'noopener'"),
      '#default_value' => $this->settings['rel_noopener'] ?? FALSE,
    ];
    $form['rel_noreferrer'] = [
      '#type' => 'checkbox',
      '#title' => $this->t("Add rel attribute 'noreferrer'"),
      '#default_value' => $this->settings['rel_noreferrer'] ?? FALSE,
    ];
    $form['css_classes'] = [
      '#type' => 'textfield',
      '#title' => $this->t('CSS classes to add to the link element'),
      '#default_value' => $this->settings['css_classes'],
      '#description' => $this->t(
        'Space separated list of CSS classes to add to the link element for enhanced display customization.'
      ),
    ];
    $form['html_suffix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom HTML suffix'),
      '#default_value' => $this->settings['html_suffix'],
      '#description' => $this->t('Custom HTML code to append to the link text inside the link element.'),
    ];
    return $form;
  }

}
