(function (Drupal) {
  'use strict';

  // --- BEGIN: improved CSSTree linter ---

  // Quick structural balance check for {}, (), [] to catch missing/extra braces.
  function balanceCheck(source) {
    const issues = [];
    let line = 0, ch = 0;
    const stack = []; // {char, line, ch}
    const openers = { '{': '}', '(': ')', '[': ']' };
    const closers = new Set(['}', ')', ']']);

    for (let i = 0; i < source.length; i++) {
      const c = source[i];

      if (c === '\n') {
        line++;
        ch = 0;
        continue;
      }

      if (openers[c]) {
        stack.push({ char: c, line, ch });
      } else if (closers.has(c)) {
        if (!stack.length) {
          issues.push({ line, ch, message: `Unexpected "${c}"`, severity: 'error' });
        } else {
          const last = stack.pop();
          if (openers[last.char] !== c) {
            issues.push({ line, ch, message: `Mismatched "${c}" (expected "${openers[last.char]}")`, severity: 'error' });
          }
        }
      }

      ch++;
    }

    // Any unclosed structures left over.
    stack.forEach(s => {
      issues.push({ line: s.line, ch: s.ch, message: `Unclosed "${s.char}"`, severity: 'error' });
    });

    return issues;
  }

  // Lint using CSSTree for modern syntax + value validation.
  function lintWithCSSTree(source) {
    const issues = [];

    // 1) Structural balance first (fast preflight).
    issues.push(...balanceCheck(source));

    if (!window.csstree) {
      return issues;
    }

    try {
      const ast = csstree.parse(source, {
        positions: true,
        onParseError: (err) => {
          issues.push({
            line: (err.line ?? 1) - 1,
            ch: (err.column ?? 1) - 1,
            message: err.message || 'CSS parse error',
            severity: 'error',
          });
        },
      });

      // 2) Unknown properties + invalid values (skip custom props and vendor-prefixed).
      csstree.walk(ast, {
        visit: 'Declaration',
        enter(node) {
          const prop = (node.property || '').trim();

          const isCustom = prop.startsWith('--');
          const isVendor = prop.startsWith('-webkit-') || prop.startsWith('-moz-') || prop.startsWith('-ms-') || prop.startsWith('-o-');

          if (!isCustom && !isVendor) {
            const known = !!csstree.lexer.properties[prop];
            if (!known) {
              const loc = (node.loc && node.loc.start) || { line: 1, column: 1 };
              issues.push({
                line: loc.line - 1,
                ch: loc.column - 1,
                message: `Unknown property "${prop}"`,
                severity: 'warning',
              });
            }

            // Validate value against CSSTree's syntax DB.
            try {
              csstree.lexer.matchProperty(prop, node.value);
            } catch (e) {
              const vloc = (node.value && node.value.loc && node.value.loc.start)
                || (node.loc && node.loc.start)
                || { line: 1, column: 1 };
              issues.push({
                line: vloc.line - 1,
                ch: vloc.column - 1,
                message: e.message || `Invalid value for "${prop}"`,
                severity: 'error',
              });
            }
          }
        },
      });

    } catch (fatal) {
      issues.push({ line: 0, ch: 0, message: fatal.message || 'CSS parse error', severity: 'error' });
    }

    return issues;
  }

  // CodeMirror lint adapter for our CSSTree-based linter.
  function cmLintAdapter(text) {
    const list = lintWithCSSTree(text);
    return list.map((i) => ({
      from: CodeMirror.Pos(i.line, i.ch),
      to: CodeMirror.Pos(i.line, (i.ch || 0) + 1),
      message: i.message,
      severity: i.severity || 'error',
    }));
  }

  // --- END: improved CSSTree linter ---

  Drupal.behaviors.extraCssUi = {
    attach(context) {
      const areas = once('extra-css-ui', 'textarea[data-extra-css-ui="1"]', context);
      areas.forEach((ta) => {
        const editor = CodeMirror.fromTextArea(ta, {
          mode: 'text/css',
          lineNumbers: true,
          lineWrapping: true,
          matchBrackets: true,
          gutters: ['CodeMirror-lint-markers'],
          lint: { getAnnotations: cmLintAdapter, async: false },
          viewportMargin: Infinity,
        });

        // Sensible default size for admin.
        editor.setSize('100%', '480px');

        // --- Toolbar with "Format CSS" + live issue count ---
        const toolbar = document.createElement('div');
        toolbar.className = 'extra-css-ui-toolbar';
        toolbar.style.margin = '6px 0';

        const btn = document.createElement('button');
        btn.type = 'button';
        btn.className = 'button button--small';
        btn.textContent = 'Format CSS';
        btn.addEventListener('click', () => {
          if (typeof window.css_beautify === 'function') {
            const formatted = window.css_beautify(editor.getValue(), {
              indent_size: 2,
              selector_separator_newline: true,
              newline_between_rules: true,
              space_around_selector_separator: true,
              preserve_newlines: true,
            });
            editor.setValue(formatted);
          }
          editor.focus();
        });

        const errs = document.createElement('span');
        errs.style.marginLeft = '10px';
        errs.style.fontSize = '0.9em';
        errs.style.opacity = '0.8';

        function updateErrorCount() {
          const count = cmLintAdapter(editor.getValue()).length;
          errs.textContent = count ? `Issues: ${count}` : 'No issues';
        }

        editor.on('change', () => {
          // Keep the textarea in sync for Drupal form submits and AJAX.
          editor.save();
        });
        editor.on('change', updateErrorCount);
        setTimeout(updateErrorCount, 0);

        toolbar.appendChild(btn);
        toolbar.appendChild(errs);

        // Insert toolbar after the editor widget.
        const wrapper = editor.getWrapperElement();
        wrapper.parentNode.insertBefore(toolbar, wrapper.nextSibling);
      });
    },
  };

})(Drupal);
