<?php

namespace Drupal\extra_css_ui\Controller;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\CacheableResponse;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Response;

final class SiteCssController extends ControllerBase {

  /**
   * Stream CSS for any /themes/custom/{theme}/css/{filename} request.
   * Route params are not used to vary content; we always serve current config
   * CSS.
   */
  public function stream(string $theme = '', string $filename = ''): Response {
    $config = $this->config('extra_css_ui.settings');

    $css = (string) ($config->get('css') ?? '');
    $css = $this->sanitizeCss(
      $css,
      (bool) $config->get('strip_imports'),
      TRUE // block_dangerous_protocols is locked on
    );

    $maxKb = (int) ($config->get('max_kb') ?? 64);
    $maxBytes = max(1024, $maxKb * 1024);
    if (strlen($css) > $maxBytes) {
      $this->getLogger('extra_css_ui')
        ->warning('Custom CSS truncated to @kb KB.', ['@kb' => $maxKb]);
      $css = substr($css, 0, $maxBytes);
    }

    $response = new CacheableResponse($css);
    $response->headers->set('Content-Type', 'text/css; charset=UTF-8');

    // Cache forever, but tie to config so saves invalidate.
    $meta = (new CacheableMetadata())
      ->setCacheMaxAge(Cache::PERMANENT)
      ->setCacheTags(['config:extra_css_ui.settings']);
    $response->addCacheableDependency($meta);

    return $response;
  }

  /**
   * Conservative CSS sanitizer for admin-provided overrides.
   */
  private function sanitizeCss(string $css, bool $strip_imports, bool $block_protocols): string {
    $css = strip_tags($css);
    $css = str_replace(["\r\n", "\r"], "\n", $css);

    $patterns = ['#@charset\s+[^;]+;#iu'];
    if ($strip_imports) {
      $patterns[] = '#@import\s+[^;]+;#iu';
    }
    $css = preg_replace($patterns, '', $css) ?? $css;

    $css = preg_replace('#expression\s*\(#iu', '', $css) ?? $css;
    $css = str_replace('<', '', $css);

    if ($block_protocols) {
      $css = preg_replace_callback('#url\(\s*([\'"]?)([^)\'"]+)\1\s*\)#iu', function($m) {
        $url = trim($m[2]);
        $safe = UrlHelper::stripDangerousProtocols($url);
        if (preg_match('#^(data:|javascript:|vbscript:)#i', $safe)) {
          return 'url(about:blank)';
        }
        return 'url(' . $safe . ')';
      }, $css) ?? $css;
    }

    return $css;
  }

}
