<?php

namespace Drupal\extra_css_ui\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

final class SiteCssSettingsForm extends ConfigFormBase {

  public function getFormId(): string {
    return 'extra_css_ui_settings_form';
  }

  protected function getEditableConfigNames(): array {
    return ['extra_css_ui.settings'];
  }

  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('extra_css_ui.settings');

    $theme_handler = \Drupal::service('theme_handler');
    $themes = $theme_handler->listInfo();
    $options = [];
    foreach ($themes as $machine => $extension) {
      $options[$machine] = $extension->info['name'] ?? $machine;
    }

    $default_theme = (string) $this->config('system.theme')->get('default');
    $current_theme_value = $config->get('theme_machine_name') ?: $default_theme;

    /**
     * Collapsed settings fieldset (appears ABOVE the CSS field).
     */
    $form['settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Configuration'),
      '#open' => FALSE,
      '#weight' => -10,
      '#tree' => TRUE,
    ];

    $form['settings']['theme_machine_name'] = [
      '#type' => 'select',
      '#title' => $this->t('Frontend theme'),
      '#options' => $options,
      '#default_value' => $current_theme_value,
      '#required' => TRUE,
      '#description' => $this->t('Choose the front-end theme this CSS should follow. (Defaults to the site’s current default theme.)'),
    ];

    $form['settings']['theme_library_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Theme library name'),
      '#default_value' => $config->get('theme_library_name') ?? 'global-styling',
      '#description' => $this->t('Library in your theme’s *.libraries.yml that includes the final CSS. Typically this is "global-styling" but check and change if not'),
      '#required' => TRUE,
    ];

    $form['settings']['css_filename'] = [
      '#type' => 'textfield',
      '#title' => $this->t('CSS filename'),
      '#default_value' => $config->get('css_filename') ?? 'additional-style.css',
      '#description' => $this->t('The URL filename to serve (e.g. "additional-style.css"). Must end with .css'),
      '#required' => TRUE,
    ];

    $form['settings']['strip_imports'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Block @import statements (recommended)'),
      '#default_value' => $config->get('strip_imports') ?? TRUE,
    ];

    $form['settings']['max_kb'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum size (KB)'),
      '#default_value' => $config->get('max_kb') ?? 128,
      '#min' => 1,
      '#description' => $this->t('If exceeded, the CSS will be truncated to this size.'),
      '#required' => TRUE,
    ];

    $form['css'] = [
      '#type' => 'textarea',
      '#title' => $this->t('CSS additions and overrides'),
      '#description' => $this->t('Add your additions/overrides here. This CSS will then be rendered after the rest of your theme\'s CSS.'),
      '#default_value' => $config->get('css') ?? '',
      '#rows' => 24,
      '#attributes' => [
        'spellcheck' => 'false',
        'style' => 'font-family: monospace;',
        'data-extra-css-ui' => '1',
      ],
    ];

    $form['#attached']['library'][] = 'extra_css_ui/editor';

    return parent::buildForm($form, $form_state);
  }

  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $css = (string) ($form_state->getValue('css') ?? '');
    if ($css !== strip_tags($css)) {
      $form_state->setErrorByName('css', $this->t('HTML tags are not allowed. Provide only CSS rules.'));
    }

    $theme = (string) $form_state->getValue(['settings', 'theme_machine_name']);
    $library = (string) $form_state->getValue([
      'settings',
      'theme_library_name',
    ]);
    $filename = (string) $form_state->getValue(['settings', 'css_filename']);

    if (!preg_match('/^[a-z0-9_\-]+$/', $theme)) {
      $form_state->setErrorByName('settings][theme_machine_name', $this->t('Theme machine name must contain only lowercase letters, digits, underscores, or hyphens.'));
    }

    if (!preg_match('/^[a-z0-9_\-]+$/', $library)) {
      $form_state->setErrorByName('settings][theme_library_name', $this->t('Library name must contain only lowercase letters, digits, underscores, or hyphens.'));
    }

    if (!preg_match('/^[a-zA-Z0-9._\-]+\.css$/', $filename)) {
      $form_state->setErrorByName('settings][css_filename', $this->t('Filename must end with .css and contain only letters, digits, dot, underscore, or hyphen.'));
    }

    $max_kb = (int) $form_state->getValue(['settings', 'max_kb']);
    if ($max_kb < 1) {
      $form_state->setErrorByName('settings][max_kb', $this->t('Maximum size must be at least 1 KB.'));
    }
  }

  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValue('settings') ?? [];

    $this->configFactory()->getEditable('extra_css_ui.settings')
      ->set('css', (string) $form_state->getValue('css'))
      ->set('strip_imports', (bool) ($values['strip_imports'] ?? TRUE))
      ->set('max_kb', (int) ($values['max_kb'] ?? 128))
      ->set('theme_machine_name', (string) ($values['theme_machine_name'] ?? ''))
      ->set('theme_library_name', (string) ($values['theme_library_name'] ?? 'global-styling'))
      ->set('css_filename', (string) ($values['css_filename'] ?? 'additional-style.css'))
      ->save();

    parent::submitForm($form, $form_state);
    $this->messenger()->addStatus($this->t('Settings saved.'));
  }

}