<?php

declare(strict_types=1);

namespace Drupal\facets_form_fulltext;

use Drupal\facets\FacetInterface;

/**
 * Represents a fulltext data model.
 */
final class Fulltext {

  /**
   * Searched text.
   *
   * @var string|null
   */
  protected $search = NULL;

  /**
   * Returns the searched text.
   *
   * @return string|null
   *   Searched text.
   */
  public function getSearch(): ?string {
    return $this->search;
  }

  /**
   * Sets the search.
   *
   * @param string|null $search
   *   Searched text.
   *
   * @return $this
   */
  public function setSearch(?string $search): self {
    $this->search = trim($search ?? '');
    return $this;
  }

  /**
   * Checks search is empty.
   *
   * @return bool
   *   The search is empty.
   */
  public function isEmpty(): bool {
    return empty($this->getSearch());
  }

  /**
   * Creates a full text from a given facet active items.
   *
   * @param \Drupal\facets\FacetInterface $facet
   *   The facet.
   *
   * @return self
   *   A Fulltext object.
   */
  public static function createFromFacet(FacetInterface $facet): Fulltext {
    $full_text = new static();
    if ($active_items = $facet->getActiveItems()) {
      $full_text->setSearch($active_items[0]);
    }
    return $full_text;
  }

  /**
   * Turn to string.
   */
  public function __toString(): string {
    return $this->getSearch();
  }

}
