<?php

declare(strict_types=1);

namespace Drupal\facets_form_fulltext\Plugin\facets\query_type;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\facets\QueryType\QueryTypePluginBase;
use Drupal\facets\Result\Result;
use Drupal\facets_form_fulltext\Fulltext;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Query type plugin that filters the result by the text active filter.
 *
 * @FacetsQueryType(
 *   id = "facets_form_fulltext_query_type",
 *   label = @Translation("Fulltext query type (Facets Form)"),
 * )
 */
final class FulltextQueryType extends QueryTypePluginBase implements ContainerFactoryPluginInterface {

  /**
   * The fulltext object used by Widget and QueryType.
   *
   * @var \Drupal\facets_form_fulltext\Fulltext
   */
  protected $fulltext;

  /**
   * Constructs a new facet plugin instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->fulltext = Fulltext::createFromFacet($this->facet);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    if (empty($this->query)) {
      return;
    }

    if ($this->fulltext->isEmpty()) {
      return;
    }

    // Prepare the search value.
    $widget_config = $this->facet->getWidgetInstance()->getConfiguration();
    $operator = $widget_config['operator'] ?? '=';

    if ($operator === 'AND') {
      $and_condition_group = $this->query->createConditionGroup('AND');
      $search_values = preg_split('/\s+/', trim($this->fulltext->getSearch()), -1, PREG_SPLIT_NO_EMPTY);
      foreach ($search_values as $search_value) {
        $and_condition_group->addCondition($this->facet->getFieldIdentifier(), $search_value);
      }
      $this->query->addConditionGroup($and_condition_group);
    }
    else {
      $search_value = $this->fulltext->getSearch();
      $this->query->addCondition($this->facet->getFieldIdentifier(), $search_value, $operator);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $build = [];

    if ($this->fulltext->isEmpty()) {
      return $build;
    }

    $display = $this->t('Contains @search', [
      '@search' => $this->fulltext->getSearch(),
    ]);

    $build[] = new Result($this->facet, (string) $this->fulltext, $display, 0);
    $this->facet->setResults($build);

    return $build;
  }

}
