<?php

declare(strict_types=1);

namespace Drupal\facets_form_fulltext\Plugin\facets\widget;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\facets\FacetInterface;
use Drupal\facets\Plugin\facets\widget\ArrayWidget;
use Drupal\facets\Result\Result;
use Drupal\facets_form\FacetsFormWidgetInterface;
use Drupal\facets_form\FacetsFormWidgetTrait;
use Drupal\facets_form_fulltext\Fulltext;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Fulltext widget as form element.
 *
 * @FacetsWidget(
 *   id = "facets_form_fulltext",
 *   label = @Translation("Fulltext (inside form)"),
 *   description = @Translation("A configurable widget that shows an input text as a form element."),
 * )
 */
final class FulltextWidget extends ArrayWidget implements FacetsFormWidgetInterface, ContainerFactoryPluginInterface {

  use FacetsFormWidgetTrait;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new facet plugin instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'label' => $this->t('Search'),
      'placeholder' => $this->t('Type a word'),
      'operator' => '=',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state, FacetInterface $facet): array {
    $configuration = $this->getConfiguration();

    return [
      'label' => [
        '#type' => 'textfield',
        '#title' => $this->t('Label'),
        '#default_value' => $configuration['label'],
      ],
      'placeholder' => [
        '#type' => 'textfield',
        '#title' => $this->t('Placeholder'),
        '#default_value' => $configuration['placeholder'],
      ],
      'operator' => [
        '#type' => 'select',
        '#title' => $this->t('Operator'),
        '#options' => [
          '=' => 'Equals',
          'AND' => 'And ("foo bar" or "bar foo" should return same result)',
        ],
        '#default_value' => $configuration['operator'],
      ],
    ] + parent::buildConfigurationForm($form, $form_state, $facet);
  }

  /**
   * {@inheritdoc}
   */
  public function build(FacetInterface $facet) {
    $fulltext = Fulltext::createFromFacet($facet);
    $configuration = $facet->getWidgetInstance()->getConfiguration();

    if (!$facet->getResults()) {
      // Set fake results so that the build will not consider the empty
      // behavior.
      // @see \Drupal\facets\FacetManager\DefaultFacetManager::build()
      $facet->setResults([new Result($facet, NULL, NULL, 0)]);
    }

    return [
      $facet->id() => [
        '#type' => 'textfield',
        '#title' => $configuration['label'],
        '#default_value' => $fulltext->getSearch(),
        '#attributes' => [
          'placeholder' => $configuration['placeholder'],
        ],
        '#cache' => [
          'contexts' => [
            'url.query_args',
            'url.path',
          ],
        ],
        '#attached' => [
          'library' => [
            'facets_form_fulltext/facets_form_fulltext',
          ],
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function prepareValueForUrl(FacetInterface $facet, array &$form, FormStateInterface $form_state): array {
    $fulltext = (new Fulltext())
      ->setSearch($form_state->getValue($facet->id()));
    return !$fulltext->isEmpty() ? [(string) $fulltext] : [];
  }

  /**
   * {@inheritdoc}
   */
  public function getQueryType(): string {
    return 'facets_form_fulltext';
  }

}
