<?php

declare(strict_types=1);

namespace Drupal\Tests\facets_form_fulltext\Kernel\Plugin\facets\widget;

use Drupal\Core\Form\FormState;
use Drupal\facets\Entity\Facet;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\facets_form\Traits\FacetsFormWidgetTestTrait;

/**
 * Tests for the Fulltext Widget functionality.
 *
 * This test class covers all aspects of the fulltext search widget:
 * - Basic rendering and structure.
 * - Form submission and URL preparation.
 * - Configuration management.
 * - Edge cases and validation.
 *
 * @coversDefaultClass \Drupal\facets_form_fulltext\Plugin\facets\widget\FulltextWidget
 * @group facets_form_fulltext
 *
 * @SuppressWarnings(PHPMD.TooManyPublicMethods)
 */
class FulltextWidgetTest extends KernelTestBase {

  use FacetsFormWidgetTestTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'facets',
    'facets_form',
    'facets_form_test',
    'facets_form_fulltext',
  ];

  // =========================================================================
  // BASIC WIDGET RENDERING TESTS
  // =========================================================================

  /**
   * Tests that the widget renders correctly with no search terms.
   *
   * Verifies:
   * - Widget creates a textfield element.
   * - Default value is empty when no active search.
   * - Placeholder attribute is properly set.
   */
  public function testWithEmptyItems(): void {
    $facet = new Facet(['id' => 'foo'], 'facets_facet');
    $facet->setFacetSourceId('facets_form_test');
    $facet->setWidget('facets_form_fulltext');

    $build = $facet->getWidgetInstance()->build($facet);

    $this->assertNotEmpty($build);
    $this->assertArrayHasKey('foo', $build);

    $element = $build['foo'];
    $this->assertSame('textfield', $element['#type']);
    $this->assertEmpty($element['#default_value']);
    $this->assertArrayHasKey('placeholder', $element['#attributes']);
  }

  /**
   * Tests that the widget always renders even when facet has no results.
   *
   * Unlike other facet widgets that might hide when empty,
   * fulltext search should always be available to users.
   */
  public function testAlwaysBuildWithEmptyResults(): void {
    $facet = new Facet(['id' => 'empty_facet'], 'facets_facet');
    $facet->setFacetSourceId('facets_form_test');
    $facet->setWidget('facets_form_fulltext');

    $facet->setResults([]);

    $build = $facet->getWidgetInstance()->build($facet);

    $this->assertNotEmpty($build);
    $this->assertArrayHasKey('empty_facet', $build);
    $this->assertEquals('textfield', $build['empty_facet']['#type']);
  }

  /**
   * Tests that active search terms populate the input field correctly.
   *
   * When a user has already performed a search, the search term
   * should appear in the input field for editing/refinement.
   */
  public function testSearchValueFromActiveItems(): void {
    $facet = new Facet(['id' => 'foo'], 'facets_facet');
    $facet->setFacetSourceId('facets_form_test');
    $facet->setWidget('facets_form_fulltext');

    $facet->setActiveItems(['my search term']);

    $build = $facet->getWidgetInstance()->build($facet)['foo'];

    $this->assertEquals('my search term', $build['#default_value']);
  }

  /**
   * Tests that facet with no active search shows empty input field.
   *
   * Baseline test to ensure clean state when no search is active.
   */
  public function testEmptySearchValue(): void {
    $facet = new Facet(['id' => 'foo'], 'facets_facet');
    $facet->setFacetSourceId('facets_form_test');
    $facet->setWidget('facets_form_fulltext');

    $build = $facet->getWidgetInstance()->build($facet)['foo'];

    $this->assertEmpty($build['#default_value']);
  }

  // =========================================================================
  // FORM SUBMISSION AND URL PREPARATION TESTS
  // =========================================================================

  /**
   * Tests successful form submission with valid search term.
   *
   * When user submits a search, the widget should properly
   * prepare the value for URL generation.
   */
  public function testPrepareValueForUrl(): void {
    $facet = new Facet(['id' => 'search_field'], 'facets_facet');
    $facet->setFacetSourceId('facets_form_test');
    $facet->setWidget('facets_form_fulltext');

    $form = [];
    $form_state = new FormState();
    $form_state->setValue('search_field', 'user typed this');

    $result = $facet->getWidgetInstance()->prepareValueForUrl($facet, $form, $form_state);

    $this->assertCount(1, $result);
    $this->assertEquals('user typed this', $result[0]);
  }

  /**
   * Tests that empty form submission doesn't add URL parameters.
   *
   * Empty searches should not pollute the URL with empty parameters.
   */
  public function testPrepareValueForUrlEmpty(): void {
    $facet = new Facet(['id' => 'search_field'], 'facets_facet');
    $facet->setFacetSourceId('facets_form_test');
    $facet->setWidget('facets_form_fulltext');

    $form = [];
    $form_state = new FormState();
    $form_state->setValue('search_field', '');

    $result = $facet->getWidgetInstance()->prepareValueForUrl($facet, $form, $form_state);

    $this->assertEmpty($result);
  }

  /**
   * Tests that whitespace-only input is treated as empty.
   *
   * User might accidentally submit spaces - these should be
   * trimmed and treated as no search term.
   */
  public function testPrepareValueForUrlWhitespace(): void {
    $facet = new Facet(['id' => 'search_field'], 'facets_facet');
    $facet->setFacetSourceId('facets_form_test');
    $facet->setWidget('facets_form_fulltext');

    $form = [];
    $form_state = new FormState();
    $form_state->setValue('search_field', '   ');

    $result = $facet->getWidgetInstance()->prepareValueForUrl($facet, $form, $form_state);

    $this->assertEmpty($result);
  }

  // =========================================================================
  // COMPREHENSIVE WIDGET BEHAVIOR TESTS
  // =========================================================================

  /**
   * Tests comprehensive widget behavior across different configurations.
   *
   * This parameterized test covers:
   * - Default vs custom labels and placeholders.
   * - Active vs inactive search states.
   * - Multiple active items handling.
   *
   * @param array $facet_values
   *   Base facet entity configuration.
   * @param array $widget_config
   *   Widget-specific configuration overrides.
   * @param array $active_items
   *   Currently active search terms.
   * @param string $expected_title
   *   Expected form element title.
   * @param string $expected_placeholder
   *   Expected placeholder text.
   * @param string $expected_default_value
   *   Expected pre-filled value.
   *
   * @dataProvider providerTestPlugin
   */
  public function testPlugin(
    array $facet_values,
    array $widget_config,
    array $active_items,
    string $expected_title,
    string $expected_placeholder,
    string $expected_default_value,
  ): void {
    $facet = new Facet(['id' => 'foo'] + $facet_values, 'facets_facet');
    $facet->setFacetSourceId('facets_form_test');
    $facet->setWidget('facets_form_fulltext', $widget_config);

    if (!empty($active_items)) {
      $facet->setActiveItems($active_items);
    }

    $build = $facet->getWidgetInstance()->build($facet)['foo'];

    $this->assertEquals($expected_title, $build['#title']);
    $this->assertEquals($expected_placeholder, $build['#attributes']['placeholder']);
    $this->assertEquals($expected_default_value, $build['#default_value']);
  }

  /**
   * Data provider for comprehensive widget testing.
   *
   * Each scenario tests different aspects of widget configuration
   * and state management.
   */
  public static function providerTestPlugin(): array {
    return [
      'clean_state_default_config' => [
        'facet_values' => [],
        'widget_config' => [],
        'active_items' => [],
        'expected_title' => 'Search',
        'expected_placeholder' => 'Type a word',
        'expected_default_value' => '',
      ],
      'active_search_default_config' => [
        'facet_values' => [],
        'widget_config' => [],
        'active_items' => ['test search'],
        'expected_title' => 'Search',
        'expected_placeholder' => 'Type a word',
        'expected_default_value' => 'test search',
      ],
      'custom_labels_with_search' => [
        'facet_values' => [],
        'widget_config' => [
          'label' => 'Find articles',
          'placeholder' => 'Enter keywords...',
        ],
        'active_items' => ['article search'],
        'expected_title' => 'Find articles',
        'expected_placeholder' => 'Enter keywords...',
        'expected_default_value' => 'article search',
      ],
      'multiple_terms_uses_first_only' => [
        'facet_values' => [],
        'widget_config' => [],
        'active_items' => ['first term', 'second term'],
        'expected_title' => 'Search',
        'expected_placeholder' => 'Type a word',
        'expected_default_value' => 'first term',
      ],
    ];
  }

  // =========================================================================
  // QUERY TYPE AND TECHNICAL INTEGRATION TESTS
  // =========================================================================

  /**
   * Tests that widget reports correct query type for Search API integration.
   *
   * The query type determines how Search API processes the search terms.
   */
  public function testGetQueryType(): void {
    $facet = new Facet(['id' => 'foo'], 'facets_facet');
    $facet->setFacetSourceId('facets_form_test');
    $facet->setWidget('facets_form_fulltext');

    $this->assertEquals('facets_form_fulltext', $facet->getWidgetInstance()->getQueryType());
  }

  /**
   * Tests that widget output includes all required technical elements.
   *
   * Verifies:
   * - Proper cache contexts for URL dependencies.
   * - Required JavaScript libraries are attached.
   * - Form structure is correct.
   */
  public function testBuildStructure(): void {
    $facet = new Facet(['id' => 'test_facet'], 'facets_facet');
    $facet->setFacetSourceId('facets_form_test');
    $facet->setWidget('facets_form_fulltext');

    $build = $facet->getWidgetInstance()->build($facet);

    $this->assertArrayHasKey('test_facet', $build);
    $element = $build['test_facet'];

    // Form structure validation.
    $this->assertEquals('textfield', $element['#type']);

    // Cache and performance validation.
    $this->assertArrayHasKey('#cache', $element);
    $this->assertContains('url.query_args', $element['#cache']['contexts']);
    $this->assertContains('url.path', $element['#cache']['contexts']);

    // Frontend assets validation.
    $this->assertArrayHasKey('#attached', $element);
    $this->assertContains('facets_form_fulltext/facets_form_fulltext', $element['#attached']['library']);
  }

  // =========================================================================
  // CONFIGURATION MANAGEMENT TESTS
  // =========================================================================

  /**
   * Tests that admin configuration form contains all required fields.
   *
   * Administrators should be able to configure:
   * - Widget label (what users see)
   * - Placeholder text (input hint)
   * - Search operator (how terms are matched)
   */
  public function testConfigurationForm(): void {
    $facet = new Facet(['id' => 'foo'], 'facets_facet');
    $facet->setFacetSourceId('facets_form_test');
    $facet->setWidget('facets_form_fulltext');

    $form = [];
    $form_state = new FormState();

    $config_form = $facet->getWidgetInstance()->buildConfigurationForm($form, $form_state, $facet);

    // Verify all configuration fields exist.
    $this->assertArrayHasKey('label', $config_form);
    $this->assertArrayHasKey('placeholder', $config_form);
    $this->assertArrayHasKey('operator', $config_form);

    // Verify field types are appropriate.
    $this->assertEquals('textfield', $config_form['label']['#type']);
    $this->assertEquals('textfield', $config_form['placeholder']['#type']);
    $this->assertEquals('select', $config_form['operator']['#type']);

    // Verify operator options are available.
    $this->assertArrayHasKey('=', $config_form['operator']['#options']);
    $this->assertArrayHasKey('AND', $config_form['operator']['#options']);
    $this->assertEquals('=', $config_form['operator']['#default_value']);
  }

  /**
   * Tests that widget provides sensible default configuration values.
   *
   * New widgets should work out-of-the-box with reasonable defaults.
   */
  public function testDefaultConfiguration(): void {
    $facet = new Facet(['id' => 'foo'], 'facets_facet');
    $facet->setFacetSourceId('facets_form_test');
    $facet->setWidget('facets_form_fulltext');

    $default_config = $facet->getWidgetInstance()->defaultConfiguration();

    $this->assertArrayHasKey('label', $default_config);
    $this->assertArrayHasKey('placeholder', $default_config);
    $this->assertArrayHasKey('operator', $default_config);

    // Verify translatable strings are properly configured.
    $this->assertEquals('Search', $default_config['label']->getUntranslatedString());
    $this->assertEquals('Type a word', $default_config['placeholder']->getUntranslatedString());

    // Verify sensible operator default.
    $this->assertEquals('=', $default_config['operator']);
  }

}
