<?php

declare(strict_types=1);

namespace Drupal\facets_max_facets\Plugin\facets\processor;

use Drupal\facets\FacetInterface;
use Drupal\facets\Processor\BuildProcessorInterface;
use Drupal\facets\Processor\ProcessorPluginBase;
use Drupal\facets\Result\ResultInterface;

/**
 * A facets processor that prevents facet rendering when over site max facets.
 *
 * @FacetsProcessor(
 *   id = "respect_global_max_facets",
 *   label = @Translation("Respect global max facets"),
 *   description = @Translation("Prevents adding new facet selections once the global max is reached."),
 *   stages = {
 *     "build" = 100
 *   }
 * )
 */
final class RespectGlobalMaxFacets extends ProcessorPluginBase implements BuildProcessorInterface {

  /**
   * {@inheritdoc}
   */
  public function build(FacetInterface $facet, array $results): array {
    // Per-facet opt-in.
    if (!isset($facet->getProcessorConfigs()['respect_global_max_facets'])) {
      return $results;
    }

    // Global max.
    $max = (int) \Drupal::config('facets_max_facets.settings')
      ->get('max_active_facets');

    if ($max <= 0) {
      return $results;
    }

    // Count total active facets from request (?f[]).
    $query = \Drupal::request()->query->all();

    if (!isset($query['f'])) {
      return $results;
    }
    $active = $query['f'];
    $total_active = is_array($active) ? count(array_filter($active)) : 0;

    if ($total_active < $max) {
      return $results;
    }

    // Enforce: keep only active branches so users can remove filters.
    $filtered = [];

    foreach ($results as $result) {
      if ($result instanceof ResultInterface && $this->hasActiveDescendant($result)) {
        $filtered[] = $this->pruneInactiveBranches($result);
      }
    }

    static $message_shown = FALSE;

    if (!$message_shown) {
      $config = \Drupal::config('facets_max_facets.settings');
      $message = $config->get('limit_message');

      if (is_string($message) && $message !== '') {
        $message = str_replace(':max-count', (string) $max, $message);
        \Drupal::messenger()->addWarning($message);
        $message_shown = TRUE;
      }
    }

    return $filtered;
  }

  /**
   * Determines whether this result or any of its descendants is active.
   */
  private function hasActiveDescendant(ResultInterface $result): bool {
    if ($result->isActive()) {
      return TRUE;
    }

    foreach ($result->getChildren() as $child) {
      if ($child instanceof ResultInterface && $this->hasActiveDescendant($child)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Removes inactive branches while preserving active paths.
   */
  private function pruneInactiveBranches(ResultInterface $result): ResultInterface {
    $children = $result->getChildren();
    $kept_children = [];

    foreach ($children as $child) {
      if ($child instanceof ResultInterface && $this->hasActiveDescendant($child)) {
        $kept_children[] = $this->pruneInactiveBranches($child);
      }
    }

    $result->setChildren($kept_children);

    return $result;
  }

}
