<?php

namespace Drupal\facets_navlinks_widget\Plugin\facets\widget;

use Drupal\Core\Form\FormStateInterface;
use Drupal\facets\FacetInterface;
use Drupal\facets\Result\Result;
use Drupal\facets\Result\ResultInterface;
use Drupal\facets\Widget\WidgetPluginBase;

/**
 * Provides a widget to display facets as navigation links.
 *
 * @FacetsWidget(
 *   id = "navlinks",
 *   label = @Translation("List of navigation links"),
 *   description = @Translation("Display facets as a list of navigation links"),
 * )
 */
class NavLinksWidget extends WidgetPluginBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'show_reset_link' => FALSE,
      'hide_reset_when_no_selection' => FALSE,
      'reset_text' => $this->t('Show all'),
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state, FacetInterface $facet) {
    $form = parent::buildConfigurationForm($form, $form_state, $facet);

    $form['show_reset_link'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show reset link'),
      '#default_value' => $this->getConfiguration()['show_reset_link'],
    ];
    $form['reset_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Reset text'),
      '#default_value' => $this->getConfiguration()['reset_text'],
      '#states' => [
        'visible' => [
          ':input[name="widget_config[show_reset_link]"]' => ['checked' => TRUE],
        ],
        'required' => [
          ':input[name="widget_config[show_reset_link]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['hide_reset_when_no_selection'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Hide reset link when no facet item is selected'),
      '#default_value' => $this->getConfiguration()['hide_reset_when_no_selection'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function build(FacetInterface $facet) {
    $build = parent::build($facet);

    // Process the active facet link to add facet filter key.
    foreach ($build['#items'] as $delta => $item) {
      if ($item['#title']['#is_active'] === TRUE) {

        $item_facet = $item['#title']['#facet'];

        $urlProcessorManager = \Drupal::service('plugin.manager.facets.url_processor');
        $url_processor = $urlProcessorManager->createInstance($item_facet->getFacetSourceConfig()->getUrlProcessorName(), ['facet' => $item_facet]);

        $request = \Drupal::request();
        $params = $request->query->all();
        $params[$url_processor->getFilterKey()] = [$item_facet->getUrlAlias() . $url_processor->getSeparator() . $item['#title']['#raw_value']];
        $build['#items'][$delta]['#url']->setOption('query', $params);
        break;
      }
    }

    $this->addResetLink($facet, $build);

    $build['#theme'] = 'facets_item_list_navlinks';

    return $build;
  }

  /**
   * Helper to add a reset facets link.
   */
  protected function addResetLink(FacetInterface $facet, array &$build) {
    $results = $facet->getResults();

    // Add reset link.
    if ($this->getConfiguration()['show_reset_link'] && count($results) > 0 && (!$this->getConfiguration()['hide_reset_when_no_selection'] || $facet->getActiveItems())) {
      $max_items = array_sum(array_map(function ($item) {
          return $item->getCount();
      }, $results));

      $urlProcessorManager = \Drupal::service('plugin.manager.facets.url_processor');
      $url_processor = $urlProcessorManager->createInstance($facet->getFacetSourceConfig()->getUrlProcessorName(), ['facet' => $facet]);
      $active_filters = $url_processor->getActiveFilters();

      unset($active_filters[$facet->id()]);

      $urlGenerator = \Drupal::service('facets.utility.url_generator');
      if ($active_filters) {
        $url = $urlGenerator->getUrl($active_filters, FALSE);
      }
      else {
        $request = \Drupal::request();
        $facet_source = $facet->getFacetSource();
        $url = $urlGenerator->getUrlForRequest($request, $facet_source ? $facet_source->getPath() : NULL);
        $params = $request->query->all();
        unset($params[$url_processor->getFilterKey()]);
        if (\array_key_exists('page', $params)) {
          // Go back to the first page on reset.
          unset($params['page']);
        }
        $url->setRouteParameter('facets_query', '');
        $url->setOption('query', $params);
      }

      $result_item = new Result($facet, 'reset_all', $this->getConfiguration()['reset_text'], $max_items);
      $result_item->setActiveState(FALSE);
      $result_item->setUrl($url);

      // Check if any other facet is in use.
      $none_active = TRUE;
      foreach ($results as $result) {
        if ($result->isActive() || $result->hasActiveChildren()) {
          $none_active = FALSE;
          break;
        }
      }

      // Add an is-active class when no other facet is in use.
      if ($none_active) {
        $result_item->setActiveState(TRUE);
      }

      // Build item.
      $item = $this->buildListItems($facet, $result_item);

      // Add a class for the reset link wrapper.
      $item['#wrapper_attributes']['class'][] = 'facets-reset';

      // Put reset facet link on first place.
      array_unshift($build['#items'], $item);
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function buildListItems(FacetInterface $facet, ResultInterface $result) {
    $items = parent::buildListItems($facet, $result);
    $items['#attributes']['data-drupal-facet-widget-element-class'] = 'facets-navlink';
    return $items;
  }

  /**
   * {@inheritdoc}
   */
  protected function buildResultItem(ResultInterface $result) {
    $build = parent::buildResultItem($result);
    $build['#theme'] = 'facets_result_item_navlinks';
    return $build;
  }

}
