<?php

declare(strict_types=1);

namespace Drupal\farm_quick\Traits;

use Drupal\Core\Entity\EntityMalformedException;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\asset\Entity\Asset;

/**
 * Provides methods for working with assets.
 */
trait QuickAssetTrait {

  use MessengerTrait;
  use StringTranslationTrait;
  use QuickStringTrait;

  /**
   * Returns the quick form ID.
   *
   * This must be implemented by the quick form class that uses this trait.
   *
   * @see \Drupal\farm_quick\Plugin\QuickForm\QuickFormInterface
   *
   * @return string
   *   The quick form ID.
   */
  abstract public function getQuickId();

  /**
   * Create an asset.
   *
   * @param array $values
   *   An array of values to initialize the asset with.
   *
   * @return \Drupal\asset\Entity\AssetInterface
   *   The asset entity that was created.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\Core\Entity\EntityMalformedException
   */
  protected function createAsset(array $values = []) {

    // Trim the asset name to 255 characters.
    if (!empty($values['name'])) {
      $values['name'] = $this->trimString($values['name'], 255);
    }

    // Start a new asset entity with the provided values.
    /** @var \Drupal\asset\Entity\AssetInterface $asset */
    $asset = Asset::create($values);

    // Track which quick form created the entity.
    $asset->get('quick')->appendItem($this->getQuickId());

    // Validate the entity and throw an exception if there are violations.
    $violations = $asset->validate();
    if ($violations->count() > 0) {
      throw new EntityMalformedException('The asset generated by quick form ' . $this->getQuickId() . ' failed validation.');
    }

    // Save the asset.
    $asset->save();

    // Display a message with a link to the asset.
    $message = $this->t('Asset created: <a href=":url">@name</a>', [':url' => $asset->toUrl()->toString(), '@name' => $asset->label()]);
    $this->messenger->addStatus($message);

    // Return the asset entity.
    return $asset;
  }

}
