<?php

namespace Drupal\farm_multitenant\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Defines access logic for entities based on group membership. A user can
 * access entities that are assigned to the farm organization associated with
 * the group that the user is a member of.
 */
class AccessControlHandlerBase extends EntityAccessControlHandler {

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    $entity_type = $entity->getEntityTypeId();

    if ($account->hasPermission('administer farm_multitenant')) {
      return AccessResult::allowed();
    }

    $access = parent::checkAccess($entity, $operation, $account);
    if ($access->isForbidden()) {
      return $access;
    }

    if ($operation === 'view' || $operation === 'update' || $operation === 'delete') {
      $permissions = [
        $operation . ' any ' . $entity_type,
        $operation . ' any ' . $entity->bundle() . ' ' . $entity_type,
      ];
      foreach ($permissions as $permission) {
        if ($account->hasPermission($permission)) {
          return $this->checkGroupAccess($entity, $account)
            ->addCacheContexts(['user.permissions']);
        }
      }
    }

    return $access;
  }

  /**
   * {@inheritdoc}
   */
  public function createAccess($entity_bundle = NULL, ?AccountInterface $account = NULL, array $context = [], $return_as_object = FALSE) {
    return parent::createAccess($entity_bundle, $account, $context, $return_as_object);
  }

  /**
   * Checks if user has access through group membership.
   */
  protected function checkGroupAccess(EntityInterface $entity, AccountInterface $account) {
    $entity_type = $entity->getEntityTypeId();

    // Load the farm organization referenced by the entity.
    if (!$entity->hasField('farm') || $entity->get('farm')->isEmpty()) {
      return AccessResult::forbidden();
    }

    $organization = $entity->get('farm')->entity;
    if (!$organization) {
      return AccessResult::forbidden();
    }

    // Find all groups that reference this organization.
    $group_relations = \Drupal::entityTypeManager()
      ->getStorage('group_relationship')
      ->loadByProperties([
        'plugin_id' => 'group_organization:farm',
        'entity_id' => $organization->id(),
      ]);

    if (empty($group_relations)) {
      return AccessResult::forbidden();
    }

    // Check if user is member of any of these groups.
    foreach ($group_relations as $group_relation) {
      $group = $group_relation->getGroup();
      if ($group->getMember($account)) {
        return AccessResult::allowed();
      }
    }

    return AccessResult::forbidden();
  }
}
