<?php

namespace Drupal\farm_multitenant\Plugin\views\filter;

Use Drupal\farm_multitenant\Service\OrganizationService;
use Drupal\user\Entity\User;
use Drupal\views\Plugin\views\filter\FilterPluginBase;
use Drupal\views\ViewExecutable;
use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Filters content based on the user's group membership. A user can access
 * assets that are assigned to the farm organization associated with the groups
 * that the user is a member of.
 *
 * @ViewsFilter("farm_multitenant_asset_access_filter")
 */
class AssetAccessFilter extends FilterPluginBase {

  /**
   * The title for the filter.
   *
   * @var string
   */
  protected $filterTitle;

  /**
   * The group helper service.
   *
   * @var \Drupal\farm_multitenant\Service\OrganizationService
   */
  protected $OrganizationService;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, OrganizationService $organization_service) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->OrganizationService = $organization_service;
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('farm_multitenant.organization_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function init(ViewExecutable $view, DisplayPluginBase $display, array &$options = NULL) {
    parent::init($view, $display, $options);
    $this->filterTitle = t('Farm asset access filter');
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    $account = \Drupal::currentUser();

    if ($account->hasPermission('administer farm_multitenant')) {
      return;
    }

    $user = User::load($account->id());
    $group_membership_service = \Drupal::service('group.membership_loader');
    $group_memberships = $group_membership_service->loadByUser($user);

    $farm_ids = [];
    foreach ($group_memberships as $group_membership) {
      $group = $group_membership->getGroup();
      $farm_ids = array_merge($farm_ids, $this->OrganizationService->getGroupOrganizationIds($group));
    }

    $farm_ids = array_unique($farm_ids);

    if (empty($farm_ids)) {
      // If no farms found, show nothing
      $this->query->addWhereExpression(0, '1 = 0');
      return;
    }

    if (!$this->query->ensureTable('asset__farm')) {
      // If a relationship to the farm organization doesn't exist, show nothing.
      $this->query->addWhereExpression(0, '1 = 0');
      return;
    }

    $this->query->addWhere(0, 'asset__farm.farm_target_id', $farm_ids, 'IN');
  }

}
