<?php

namespace Drupal\farm_multitenant\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Defines access logic for logs based on group membership.
 */
class AccessControlHandlerLog extends AccessControlHandlerBase {
  /**
   * {@inheritdoc}
   */
  public function createAccess($entity_bundle = NULL, ?AccountInterface $account = NULL, array $context = [], $return_as_object = FALSE) {
    $account = $this->prepareUser($account);
    $permission = 'create ' . $entity_bundle . ' log';
    if ($account && $account->hasPermission($permission)) {
      return $return_as_object ? AccessResult::allowed() : AccessResult::allowed()->isAllowed();
    }

    return parent::createAccess($entity_bundle, $account, $context, $return_as_object);
  }

  /**
   * {@inheritdoc}
   */
  protected function checkGroupAccess(EntityInterface $entity, AccountInterface $account) {
    // Ensure the entity is a log.
    if ($entity->getEntityTypeId() !== 'log') {
      return AccessResult::forbidden();
    }

    $assets = [];

    // Get the asset referenced by the log.
    if ($entity->hasField('asset') && !$entity->get('asset')->isEmpty()) {
      foreach ($entity->get('asset') as $item) {
        if ($item->entity) {
          $assets[] = $item->entity;
        }
      }
    }

    // Get the location asset referenced by the log.
    if ($entity->hasField('location') && !$entity->get('location')->isEmpty()) {
      foreach ($entity->get('location') as $item) {
        if ($item->entity) {
          $assets[] = $item->entity;
        }
      }
    }

    // Get the equipment asset referenced by the log.
    if ($entity->hasField('equipment') && !$entity->get('equipment')->isEmpty()) {
      foreach ($entity->get('equipment') as $item) {
        if ($item->entity) {
          $assets[] = $item->entity;
        }
      }
    }

    // If no assets are found, deny access.
    if (empty($assets)) {
      return AccessResult::forbidden();
    }

    // Delegate to the parent class to check access for the asset. If any of the
    // assets are accessible by the user, access is granted.
    foreach ($assets as $asset) {
      $result = parent::checkGroupAccess($asset, $account);
      if ($result->isAllowed()) {
        return $result;
      }
    }

    // If no assets granted access, deny access.
    return AccessResult::forbidden();
  }

}
