<?php

namespace Drupal\farm_multitenant\Plugin\views\filter;

Use Drupal\farm_multitenant\Service\OrganizationService;
use Drupal\user\Entity\User;
use Drupal\views\Plugin\views\filter\FilterPluginBase;
use Drupal\views\ViewExecutable;
use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Filters content based on the user's group membership. A user can access
 * quantities assigned to logs that are in turn assigned to assets that are in
 * the farm organization associated with the groups the user is a member of.
 *
 * @ViewsFilter("farm_multitenant_quantity_access_filter")
 */
class QuantityAccessFilter extends FilterPluginBase {

  /**
   * The title for the filter.
   *
   * @var string
   */
  protected $filterTitle;

  /**
   * The group helper service.
   *
   * @var \Drupal\farm_multitenant\Service\OrganizationService
   */
  protected $OrganizationService;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, OrganizationService $organization_service) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->OrganizationService = $organization_service;
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('farm_multitenant.organization_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function init(ViewExecutable $view, DisplayPluginBase $display, array &$options = NULL) {
    parent::init($view, $display, $options);
    $this->filterTitle = t('Farm quantity access filter');
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    $account = \Drupal::currentUser();

    if ($account->hasPermission('administer farm_multitenant')) {
      return;
    }

    $user = User::load($account->id());
    $group_membership_service = \Drupal::service('group.membership_loader');
    $group_memberships = $group_membership_service->loadByUser($user);

    $farm_ids = [];
    foreach ($group_memberships as $group_membership) {
      $group = $group_membership->getGroup();
      $farm_ids = array_merge($farm_ids, $this->OrganizationService->getGroupOrganizationIds($group));
    }

    $farm_ids = array_unique($farm_ids);

    if (empty($farm_ids)) {
      $this->query->addWhereExpression(0, '1 = 0');
      return;
    }

    $join_manager = \Drupal::service('plugin.manager.views.join');

    $or_group_id = $this->query->setWhereGroup('OR');
    $conditions_added = FALSE;

    $reference_fields = [
      'asset' => 'asset_target_id',
      'equipment' => 'equipment_target_id',
      'location' => 'location_target_id',
    ];

    $log_quantity_alias = $this->query->ensureTable('log__quantity');
    if (!$log_quantity_alias) {
      $this->query->addWhereExpression(0, '1 = 0');
      return;
    }

    // Join log table through log_quantity.
    $log_join_config = [
      'table' => 'log',
      'field' => 'id',
      'left_table' => $log_quantity_alias,
      'left_field' => 'entity_id',
      'type' => 'INNER',
    ];
    $log_join = $join_manager->createInstance('standard', $log_join_config);
    $log_alias = $this->query->addRelationship('log_quantity_log', $log_join, $log_quantity_alias);

    if (!$log_alias) {
      $this->query->addWhereExpression(0, '1 = 0');
      return;
    }

    $or_conditions = [];
    $args = [];

    foreach ($reference_fields as $field => $field_column) {
      $log_field_table = 'log__' . $field;

      // Join reference field table to log.
      $field_join_config = [
        'table' => $log_field_table,
        'field' => 'entity_id',
        'left_table' => $log_alias,
        'left_field' => 'id',
        'type' => 'LEFT',
      ];

      $field_join = $join_manager->createInstance('standard', $field_join_config);
      $field_table_alias = $this->query->addRelationship($log_field_table . '_' . $field, $field_join, $log_alias);

      if (!$field_table_alias) {
        continue;
      }

      $asset_farm_alias = 'asset_farm_' . $field;
      $asset_join_config = [
        'type' => 'LEFT',
        'table' => 'asset__farm',
        'field' => 'entity_id',
        'left_table' => $field_table_alias,
        'left_field' => $field_column,
        'operator' => '=',
      ];

      $asset_join = $join_manager->createInstance('standard', $asset_join_config);
      $this->query->addRelationship($asset_farm_alias, $asset_join, $field_table_alias);

      if (!$this->query->ensureTable($asset_farm_alias)) {
        continue;
      }

      $placeholder = ':farm_ids_' . $field . '[]';

      $condition = "($asset_farm_alias.farm_target_id IN ($placeholder) AND $field_table_alias.$field_column IS NOT NULL)";
      $or_conditions[] = $condition;
      $args[$placeholder] = $farm_ids;

      $conditions_added = TRUE;
    }

    if ($conditions_added) {
      $where_expression = implode(' OR ', $or_conditions);

      $this->query->addWhereExpression($or_group_id, $where_expression, $args);
    } else {
      $this->query->addWhereExpression(0, '1 = 0');
    }
  }
}
