<?php

namespace Drupal\farm_multitenant\Service;

use Drupal\group\Entity\Group;
use Drupal\group\Entity\GroupInterface;
use Drupal\group\Entity\GroupRelationship;
use Drupal\organization\Entity\OrganizationInterface;

/**
 * Provides helper functions for organizations.
 */
class OrganizationService {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new OrganizationGroupManager.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct($entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Retrieves the group related to the given organization.
   *
   * @param \Drupal\organization\Entity\OrganizationInterface $organization
   *   The organization entity.
   *
   * @return \Drupal\group\Entity\Group|null
   *   The related group entity or NULL if not found.
   */
  public function getGroupFromOrganization(OrganizationInterface $organization): ?Group {
    $relations = $this->entityTypeManager
      ->getStorage('group_relationship')
      ->loadByProperties([
        'plugin_id' => 'group_organization:farm',
        'entity_id' => $organization->id(),
      ]);

    return $relations ? reset($relations)->getGroup() : NULL;
  }

  /**
   * Get the Organization IDs associated with a specific group.
   */
  public function getGroupOrganizationIds(GroupInterface $group) {
    $farm_ids = [];

    // Load group relationships for the given group.
    $relationship_ids = \Drupal::entityQuery('group_relationship')
      ->accessCheck(TRUE)
      ->condition('gid', $group->id())
      ->condition('type', 'farm-group_organization-farm')
      ->execute();

    if ($relationship_ids) {
      foreach (GroupRelationship::loadMultiple($relationship_ids) as $relationship) {
        $farm_entity = $relationship->getEntity();
        if ($farm_entity?->getEntityTypeId() === 'organization') {
          $farm_ids[] = $farm_entity->id();
        }
      }
    }

    return $farm_ids;
  }


}
