<?php

namespace Drupal\farm_multitenant\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Url;
use Drupal\farm_multitenant\Service\OrganizationService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for removing a member from an organization.
 */
class RemoveMemberActionForm extends FormBase {

  /**
   * The array of organizations the user is assigned to.
   *
   * @var \Drupal\organization\Entity\Organization[]
   */
  protected array $selectedOrganizations = [];

  /**
   * The tempstore factory.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected $tempStoreFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypemanager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The group storage.
   *
   * @var \Drupal\group\Entity\Storage\GroupStorage
   */
  protected $groupStorage;

  /**
   * The user storage.
   *
   * @var \Drupal\user\UserStorage
   */
  protected $user;

  /**
   * The organization helper service.
   *
   * @var \Drupal\farm_multitenant\Service\OrganizationService
   */
  protected $organizationHelper;

  /**
   * Constructs a RemoveMemberActionForm form object.
   *
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The tempstore factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   Current user.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    PrivateTempStoreFactory $temp_store_factory,
    EntityTypeManagerInterface $entity_type_manager,
    AccountInterface $current_user,
    MessengerInterface $messenger,
    OrganizationService $organization_group_manager,
    ) {
    $this->tempStoreFactory = $temp_store_factory;
    $this->entityTypemanager = $entity_type_manager;
    $this->groupStorage = $this->entityTypemanager->getStorage('group');
    $this->user = $this->entityTypemanager->getStorage('user');
    $this->currentUser = $current_user;
    $this->messenger = $messenger;
    $this->organizationHelper = $organization_group_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('tempstore.private'),
      $container->get('entity_type.manager'),
      $container->get('current_user'),
      $container->get('messenger'),
      $container->get('farm_multitenant.organization_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'remove_member_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $this->selectedOrganizations = $this->tempStoreFactory
      ->get('remove_member_action_organizations')
      ->get($this->currentUser->id());

    if (empty($this->selectedOrganizations)) {
      return $this->redirect('entity.organization.collection');
    }

    foreach ($this->selectedOrganizations as $organization) {
      $items[] = $organization->label();
    }

    $form['list_heading'] = [
      '#type' => 'details',
      '#title' => $this->t('Remove Member from Organization'),
      '#open' => TRUE,
    ];

    $form['list_heading']['user'] = [
      '#type' => 'entity_autocomplete',
      '#target_type' => 'user',
      '#title' => $this->t('User'),
      '#description' => $this->t('Enter a comma-separated list of user names.'),
      '#tags' => TRUE,
      '#required' => TRUE,
    ];

    $form['list_heading']['organizations'] = [
      '#title' => $this->t('Users will be removed from the following organizations'),
      '#theme' => 'item_list',
      '#items' => $items,
    ];

    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Remove member(s)'),
      '#button_type' => 'primary',
    ];
    $form['actions']['cancel_url'] = [
      '#type' => 'submit',
      '#value' => $this->t('Cancel'),
      '#button_type' => 'button',
      '#limit_validation_errors' => [],
      '#submit' => ['::cancelForm'],
    ];

    return $form;
  }

  /**
   * Batch operations.
   *
   * Processes a single pending task as part of a batch operation.
   *
   * @param object $user
   *   User object.
   * @param object $group
   *   Group object.
   * @param array|\ArrayAccess $context
   *   The context of the current batch, as defined in the @link batch Batch
   *   operations @endlink documentation.
   */
  public function batchRemoveMember($user, $group, &$context) {

    if ($group->getMember($user)) {
      $group->removeMember($user);
    }

    $context['results']['count'] = !array_key_exists('count', $context['results']) ? 1 : ++$context['results']['count'];
  }

  /**
   * Finishes an "execute tasks" batch.
   *
   * @param bool $success
   *   Indicates whether the batch process was successful.
   * @param array $results
   *   Results information passed from the processing callback.
   * @param array $operations
   *   Operations array from batch.
   */
  public function batchRemoveMemberFinished($success, array $results, array $operations) {
    if ($success) {
      if ($results['count'] == 1) {
        $this->messenger->addStatus($this->t('@count Organization is processed.', ['@count' => $results['count']]));
      }
      else {
        $this->messenger->addStatus($this->t('@count Organizations are processed.', ['@count' => $results['count']]));
      }
    }
    else {
      $error_operation = reset($operations);
      $this->messenger->addError(
        $this->t('An error occurred while processing @operation with arguments : @args',
          [
            '@operation' => $error_operation[0],
            '@args' => print_r($error_operation[0], TRUE),
          ]
        )
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    if (!empty($this->selectedOrganizations)) {

      // Load the groups associated with the selected organizations.
      foreach ($this->selectedOrganizations as $organization) {
        $groups[] = $this->organizationHelper->getGroupFromOrganization($organization);
      }

      $group_users = $form_state->getValue('user');

      // Remove users from groups using bulk action.
      foreach ($group_users as $group_user) {
        $user = $this->user->load($group_user['target_id']);

        if ($groups) {
          $operations = [];
          foreach ($groups as $group) {
            $operations[] = [
              [$this, 'batchRemoveMember'],
              [
                $user,
                $group,
              ],
            ];
          }
          $batch = [
            'title' => $this->t('Removing selected user(s) from Organizations'),
            'operations' => $operations,
            'finished' => [$this, 'batchRemoveMemberFinished'],
          ];

          batch_set($batch);
        }
      }
      $this->tempStoreFactory->get('remove_member_action_organizations')->delete($this->currentUser->id());
    }
    $form_state->setRedirect('entity.organization.collection');
  }

  /**
   * {@inheritdoc}
   */
  public function cancelForm() {
    return new Url('entity.organization.collection');
  }

}
