<?php

namespace Drupal\farm_multitenant\Plugin\Action;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Action\ActionBase;
use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\farm_multitenant\Service\OrganizationService;
use Symfony\Component\DependencyInjection\ContainerInterface;

#[Action(
  id: 'farm_remove_member',
  label: new TranslatableMarkup('Remove member from organization'),
  confirm_form_route_name: 'farm_multitenant.organization.remove_member_action_form',
  type: 'organization'
)]
class RemoveMemberAction extends ActionBase implements ContainerFactoryPluginInterface {

  /**
   * The tempstore object.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStore
   */
  protected $tempStore;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $user;

  /**
   * The organization helper service.
   *
   * @var \Drupal\farm_multitenant\Service\OrganizationService
   */
  protected $organizationHelper;

  /**
   * Constructs a new RemoveMemberAction object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The tempstore factory.
   * @param \Drupal\Core\Session\AccountInterface $user
   *   Current user.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, PrivateTempStoreFactory $temp_store_factory, AccountInterface $user, OrganizationService $organization_service) {
    $this->user = $user;
    $this->tempStore = $temp_store_factory->get('remove_member_action_organizations');
    $this->organizationHelper = $organization_service;
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('tempstore.private'),
      $container->get('current_user'),
      $container->get('farm_multitenant.organization_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function executeMultiple(array $entities) {
    $info = [];
    foreach ($entities as $organization) {
      $info[$organization->id()] = $organization;
    }
    $this->tempStore->set($this->user->id(), $info);
  }

  /**
   * {@inheritdoc}
   */
  public function execute($object = NULL) {
    $this->executeMultiple([$object]);
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, AccountInterface $account = NULL, $return_as_object = FALSE) {
    $group = $this->organizationHelper->getGroupFromOrganization($object);
    $hasPermission = $account && $group->hasPermission('administer members', $account);
    $result = $hasPermission ? AccessResult::allowed() : AccessResult::forbidden();
    return $return_as_object ? $result : $result->isAllowed();
  }

}
