<?php

namespace Drupal\farm_multitenant\Plugin\views\filter;

Use Drupal\farm_multitenant\Service\OrganizationService;
use Drupal\user\Entity\User;
use Drupal\views\Plugin\views\filter\FilterPluginBase;
use Drupal\views\ViewExecutable;
use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Filters content based on the user's group membership. A user can access logs
 * assigned to assets that are assigned to the farm organization associated with
 * the groups that the user is a member of.
 *
 * @ViewsFilter("farm_multitenant_log_access_filter")
 */
class LogAccessFilter extends FilterPluginBase {

  /**
   * The title for the filter.
   *
   * @var string
   */
  protected $filterTitle;

  /**
   * The group helper service.
   *
   * @var \Drupal\farm_multitenant\Service\OrganizationService
   */
  protected $OrganizationService;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, OrganizationService $organization_service) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->OrganizationService = $organization_service;
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('farm_multitenant.organization_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function init(ViewExecutable $view, DisplayPluginBase $display, array &$options = NULL) {
    parent::init($view, $display, $options);
    $this->filterTitle = t('Farm log access filter');
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    $account = \Drupal::currentUser();

    if ($account->hasPermission('administer farm_multitenant')) {
      return;
    }

    $user = User::load($account->id());
    $group_membership_service = \Drupal::service('group.membership_loader');
    $group_memberships = $group_membership_service->loadByUser($user);

    $farm_ids = [];
    foreach ($group_memberships as $group_membership) {
      $group = $group_membership->getGroup();
      if ($group) {
        $farm_ids = array_merge($farm_ids, $this->OrganizationService->getGroupOrganizationIds($group));
      }
    }

    $farm_ids = array_unique($farm_ids);

    if (empty($farm_ids)) {
      $this->query->addWhereExpression(0, '1 = 0');
      return;
    }

    $join_manager = \Drupal::service('plugin.manager.views.join');

    $or_group_id = $this->query->setWhereGroup('OR');
    $conditions_added = FALSE;

    // Define reference fields to check.
    $reference_fields = [
      'asset' => 'asset_target_id',
      'equipment' => 'equipment_target_id',
      'location' => 'location_target_id',
    ];

    $or_conditions = [];
    $args = [];

    foreach ($reference_fields as $field => $field_column) {
      $log_field_table = 'log__' . $field;

      if (!$this->query->ensureTable($log_field_table)) {
        continue;
      }

      $asset_farm_alias = 'asset_farm_' . $field;
      $asset_join_config = [
        'type' => 'LEFT',
        'table' => 'asset__farm',
        'field' => 'entity_id',
        'left_table' => $log_field_table,
        'left_field' => $field_column,
        'operator' => '=',
      ];

      $asset_join = $join_manager->createInstance('standard', $asset_join_config);
      $this->query->addRelationship($asset_farm_alias, $asset_join, $log_field_table);

      if (!$this->query->ensureTable($asset_farm_alias)) {
        continue;
      }

      $placeholder = ':farm_ids_' . $field . '[]';

      // Build condition for this reference type.
      $condition = "($asset_farm_alias.farm_target_id IN ($placeholder) AND $log_field_table.$field_column IS NOT NULL)";
      $or_conditions[] = $condition;
      $args[$placeholder] = $farm_ids;

      $conditions_added = TRUE;
    }

    if ($conditions_added) {
      // Combine conditions with OR.
      $where_expression = implode(' OR ', $or_conditions);

      // Add to the OR group.
      $this->query->addWhereExpression($or_group_id, $where_expression, $args);
    } else {
      $this->query->addWhereExpression(0, '1 = 0');
    }
  }

}
