<?php

namespace Drupal\fast_error_pages\EventSubscriber;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\EventSubscriber\DefaultExceptionHtmlSubscriber;
use Drupal\Core\EventSubscriber\MainContentViewSubscriber;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Render\HtmlResponse;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\fast_error_pages\FastErrorPageManager;
use Drupal\fast_error_pages\Plugin\FastError\FastErrorPagePluginBase;
use GuzzleHttp\ClientInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\ExceptionEvent;
use Symfony\Component\HttpKernel\Exception\HttpExceptionInterface;

class FastErrorPageHtmlExceptionSubscriber extends DefaultExceptionHtmlSubscriber {

  public function __construct(
    protected readonly CacheBackendInterface $cacheBackend,
    protected readonly ClientInterface $httpClient,
    protected FastErrorPageManager $errorManager,
    protected AccountProxyInterface $currentUser,
    protected ModuleHandlerInterface $moduleHandler,
  ) {}

  protected static function getPriority() {
    return 250;
  }

  protected function getHandledFormats() {
    return ['html'];
  }

  public function onException(ExceptionEvent $event) {
    $request = $event->getRequest();

    if (!$this->currentUser->isAnonymous() || $request->headers->has('X-Drupal-Fast-Error-Pages')) {
      return;
    }

    $exception = $event->getThrowable();

    $request->attributes->set('exception', $exception);
    $handled_formats = $this->getHandledFormats();
    $format = $request->query->get(MainContentViewSubscriber::WRAPPER_FORMAT, $request->getRequestFormat());

    $status_code = $exception instanceof HttpExceptionInterface && (empty($handled_formats) || in_array($format, $handled_formats))
      ? $exception->getStatusCode()
      : 500;

    /** @var FastErrorPagePluginBase $plugin_instance */
    $plugin_instance = $this->errorManager->hasDefinition($status_code)
      ? $this->errorManager->createInstance($status_code)
      : NULL;
    $url = $plugin_instance?->getUrl();

    if ($plugin_instance && !empty($url)) {
      $cid_parts = [$status_code];
      $this->moduleHandler->alter('fast_error_pages_cache_contexts', $cid_parts);
      $cid = implode(':', $cid_parts);

      $cached_info = $this->cacheBackend->get($cid);

      // Get the page and force a backend fetch if we have no cached data.
      $response = $this->getPage($request->getHost(), $url, !$cached_info);

      if (!$cached_info) {
        $cached_info = $this->cacheBackend->get($cid);
      }

      if (!$response || !$cached_info || empty($cached_info->data['tags']) || empty($cached_info->data['contexts'])) {
        return NULL;
      }

      $cacheable_metadata = new CacheableMetadata();
      $cacheable_metadata->addCacheTags($cached_info->data['tags']);
      $cacheable_metadata->addCacheContexts($cached_info->data['contexts']);
      $cacheable_metadata->addCacheTags(['4xx-response']);
      $cacheable_metadata->addCacheContexts(['url']);
      $response->addCacheableDependency($cacheable_metadata);

      $event->setResponse($response);
    }
  }

  private function getPage(string $host, string $url, bool $bustCache = FALSE): ?HtmlResponse {

    try {
      $response = $this->httpClient->request($bustCache ? 'POST' : 'GET', $url, [
        'headers' => [
          'Host' => $host,
          'X-Drupal-Fast-Error-Pages' => '1',
        ],
      ]);

      $html = $response->getBody()->getContents();

      return new HtmlResponse($html, $response->getStatusCode());
    }
    catch (\Exception $e) {
      return NULL;
    }
  }

}
