<?php

namespace Drupal\fast_revision_purge\Commands;

use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Drupal\fast_revision_purge\Service\Planner;
use Drupal\fast_revision_purge\Service\Purger;
use Drupal\fast_revision_purge\Service\IndexManager;
use Drupal\fast_revision_purge\Service\RevisionTableMap;
use Drupal\fast_revision_purge\Service\StatsStorage;

/**
 * Drush commands for Fast Revision Purge.
 *
 * Provides:
 *  - fastrev:report  (plan/dry-run)
 *  - fastrev:purge   (execute deletions)
 *  - fastrev:reindex (ensure helpful DB indexes)
 *
 * Examples:
 *  drush fastrev:report --keep-last=5 --since=2024-01-01 --protect-published --per-language --keep-paragraph-last=1
 *  drush fastrev:purge --chunk=5000 --sleep-ms=50
 *  drush fastrev:reindex
 */
final class FastRevCommands extends DrushCommands {

  /**
   * @param \Drupal\fast_revision_purge\Service\Purger $purger
   *   Purger service that performs chunked deletes for node/paragraph revisions.
   * @param \Drupal\fast_revision_purge\Service\Planner $planner
   *   Planner service that populates working tables for keep/delete sets.
   * @param \Drupal\fast_revision_purge\Service\IndexManager $indexManager
   *   Ensures DB indexes needed for fast planning/purging.
   * @param \Drupal\fast_revision_purge\Service\RevisionTableMap $map
   *   Table discovery helper (node/paragraph revision & field tables).
   * @param \Drupal\fast_revision_purge\Service\StatsStorage $stats
   *   Stats storage for last dry-run estimate and purge totals.
   */
  public function __construct(
    private Purger $purger,
    private Planner $planner,
    private IndexManager $indexManager,
    private RevisionTableMap $map,
    private StatsStorage $stats,
  ) {
    parent::__construct();
  }

  /**
   * Compute and display a dry-run plan (keep/delete counts + samples).
   *
   * Also prints:
   * - Layout Builder revisions KEEP/DELETE (if working tables exist)
   * - Potential reclaimable space (from last dry run) and "Last Dry Run" / "Last Purged" as "…ago"
   *
   * Options:
   * - keep-last (int)           Keep latest N non-default node revs per node.
   * - since (string|null)       Keep revisions since YYYY-MM-DD.
   * - protect-published (bool)  Keep latest published rev per node.
   * - per-language (bool)       Partition keep-last by (nid, langcode).
   * - keep-paragraph-last (int) Keep last M paragraph revisions per entity.
   *
   * @param array $args
   *   (Unused) Positional args.
   * @param array $options
   *   {
   *     @type int         $keep-last           Default 5
   *     @type string|null $since               Default null (disabled)
   *     @type bool        $protect-published   Default false
   *     @type bool        $per-language        Default false
   *     @type int         $keep-paragraph-last Default 1
   *   }
   * @return int
   *   CLI exit code (0 on success).
   */
  #[CLI\Command(name: 'fastrev:report', aliases: ['fr:report'])]
  #[CLI\Option(name: 'keep-last', description: 'Keep latest N non-default node revs per node (or per language with --per-language).')]
  #[CLI\Option(name: 'since', description: 'Keep revisions since date (YYYY-MM-DD).')]
  #[CLI\Option(name: 'protect-published', description: 'Protect latest published rev per node.')]
  #[CLI\Option(name: 'per-language', description: 'Partition latest N by (nid, langcode).')]
  #[CLI\Option(name: 'keep-paragraph-last', description: 'Keep last M paragraph revisions per paragraph entity.')]
  public function report(array $args = [], array $options = [
    'keep-last' => 5,
    'since' => null,
    'protect-published' => false,
    'per-language' => false,
    'keep-paragraph-last' => 1,
  ]): int {
    $this->indexManager->ensureHelpfulIndexes();

    $keepLast         = (int) $options['keep-last'];
    $since            = $options['since'] ?: null;
    $protectPublished = (bool) $options['protect-published'];
    $perLanguage      = (bool) $options['per-language'];
    $keepParLast      = (int) $options['keep-paragraph-last'];

    $this->planner->plan($keepLast, $since, $protectPublished, $perLanguage, $keepParLast);

    $db = \Drupal::database();
    $schema = $db->schema();

    $nodeKeep = (int) $db->query("SELECT COUNT(*) FROM fastrev_node_keep")->fetchField();
    $nodeDel  = (int) $db->query("SELECT COUNT(*) FROM fastrev_node_delete")->fetchField();
    $parUse   = (int) $db->query("SELECT COUNT(*) FROM fastrev_par_in_use")->fetchField();
    $parDel   = (int) $db->query("SELECT COUNT(*) FROM fastrev_par_delete")->fetchField();

    $lbKeep = $schema->tableExists('fastrev_lb_keep')
      ? (int) $db->query("SELECT COUNT(*) FROM fastrev_lb_keep")->fetchField() : 0;
    $lbDel  = $schema->tableExists('fastrev_lb_delete')
      ? (int) $db->query("SELECT COUNT(*) FROM fastrev_lb_delete")->fetchField() : 0;

    $this->io()->success("Plan ready.");
    $this->io()->writeln("Node revisions: KEEP={$nodeKeep}, DELETE={$nodeDel}");
    $this->io()->writeln("Paragraph revisions: IN_USE={$parUse}, DELETE={$parDel}");
    if ($schema->tableExists('fastrev_lb_keep') || $schema->tableExists('fastrev_lb_delete')) {
      $this->io()->writeln("Layout Builder revisions: KEEP={$lbKeep}, DELETE={$lbDel}");
    }

    // Potential space + 'ago' times from stats.
    $stats = $this->stats->get();
    $df = \Drupal::service('date.formatter');

    $potentialBytes = (int) ($stats['potential_claimable_space'] ?? 0);
    $lastDryTs = (int) ($stats['last_dryrun_timestamp'] ?? 0);
    $lastPurgeTs = (int) ($stats['last_purge_timestamp'] ?? 0);

    $this->io()->newLine();
    $this->io()->writeln('Potential space that can be cleaned up: ' . \format_size($potentialBytes));
    $this->io()->writeln('Last Dry Run: ' . ($lastDryTs ? $df->formatTimeDiffSince($lastDryTs) : 'never'));
    $this->io()->writeln('Last Purged: ' . ($lastPurgeTs ? $df->formatTimeDiffSince($lastPurgeTs) : 'never'));

    // Sample ids.
    $sampleNodeDel = $db->query("SELECT vid FROM fastrev_node_delete ORDER BY vid LIMIT 10")->fetchCol();
    $sampleParDel  = $db->query("SELECT rid FROM fastrev_par_delete ORDER BY rid LIMIT 10")->fetchCol();

    if ($sampleNodeDel) {
      $this->io()->writeln("Sample node vids to delete: " . implode(', ', $sampleNodeDel));
    }
    if ($sampleParDel) {
      $this->io()->writeln("Sample paragraph rids to delete: " . implode(', ', $sampleParDel));
    }
    return 0;
  }

  /**
   * Execute the purge using chunked deletes with optional sleep.
   *
   * Also prints Layout Builder rows deleted and the estimated bytes freed
   * (as persisted in fastrev_stats by the Purger service).
   *
   * Options:
   * - chunk (int)     Delete this many revisions per batch (default 5000).
   * - sleep-ms (int)  Milliseconds to sleep between chunks (default 0).
   *
   * @param array $args
   *   (Unused) Positional args.
   * @param array $options
   *   {
   *     @type int $chunk     Default 5000
   *     @type int $sleep-ms  Default 0
   *   }
   * @return int
   *   CLI exit code (0 on success).
   */
  #[CLI\Command(name: 'fastrev:purge', aliases: ['fr:purge'])]
  #[CLI\Option(name: 'chunk', description: 'Chunk size for deletes.')]
  #[CLI\Option(name: 'sleep-ms', description: 'Sleep milliseconds between chunks.')]
  public function purge(array $args = [], array $options = ['chunk' => 5000, 'sleep-ms' => 0]): int {
    $chunk = (int) $options['chunk'];
    $sleep = (int) $options['sleep-ms'];

    $res = $this->purger->purge($chunk, $sleep);

    // Read last-run bytes freed from stats.
    $stats = $this->stats->get();
    $bytesFreed = (int) ($stats['space_freed_last_run'] ?? 0);

    $this->io()->success("Purge complete.");
    $this->io()->writeln("Node revisions deleted: {$res->nodeRevisionsDeleted}");
    $this->io()->writeln("Paragraph revisions deleted: {$res->paragraphRevisionsDeleted}");
    $this->io()->writeln("Layout Builder rows deleted: {$res->layoutBuilderRowsDeleted}");
    $this->io()->writeln('Estimated bytes freed: ' . \format_size($bytesFreed));
    return 0;
  }

  /**
   * Ensure helpful indexes exist (safe to re-run).
   *
   * @return int
   *   CLI exit code (0 on success).
   */
  #[CLI\Command(name: 'fastrev:reindex', aliases: ['fr:reindex'])]
  public function reindex(): int {
    $this->indexManager->ensureHelpfulIndexes();
    $this->io()->success('Indexes ensured.');
    return 0;
  }
}
