<?php

namespace Drupal\fast_revision_purge\Service;

use Drupal\Core\Database\Connection;
use Psr\Log\LoggerInterface;

/**
 * Purges stale Layout Builder field revisions safely.
 *
 * Strategy:
 * - Keep the "current" layout only (node_field_data.vid).
 * - Optionally keep the last N previous revisions (MySQL 8+ only).
 * - Delete in chunks to avoid long locks.
 */
class LayoutBuilderRevisionTruncator {

  /**
   * @param \Drupal\Core\Database\Connection $db
   *   Drupal DB connection.
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger channel.
   */
  public function __construct(
    private Connection $db,
    private LoggerInterface $logger,
  ) {}

  /**
   * Returns TRUE if the LB revision field table exists.
   */
  public function fieldExists(): bool {
    return $this->db->schema()->tableExists('node_revision__layout_builder__layout');
  }

  /**
   * Returns TRUE if the node_field_data table exists (joins depend on it).
   */
  private function nodeFieldDataExists(): bool {
    return $this->db->schema()->tableExists('node_field_data');
  }

  /**
   * Very rough detector for MySQL 8+ (for window function support).
   * Returns FALSE on non-MySQL drivers.
   */
  protected function mysql8OrHigher(): bool {
    try {
      // Only attempt on MySQL/MariaDB family.
      $driver = (string) $this->db->driver();
      if (stripos($driver, 'mysql') === false && stripos($driver, 'mariadb') === false) {
        return FALSE;
      }
      // MySQL 8+ (and compatible) support window functions; fails on older versions.
      $this->db->query("SELECT ROW_NUMBER() OVER () AS rn FROM (SELECT 1) t");
      return TRUE;
    }
    catch (\Throwable $e) {
      return FALSE;
    }
  }

  /**
   * PLAN: Estimate rows/bytes that would be deleted.
   *
   * @param int $keep_last
   *   Keep last N previous revisions per node/langcode (besides current).
   *   N>0 requires MySQL 8+ (window functions). If unavailable, we plan as N=0.
   *
   * @return array{rows:int, approx_bytes:int, used_keep_last:int}
   *   rows: number of rows that would be deleted from node_revision__layout_builder__layout
   *   approx_bytes: estimate using AVG_ROW_LENGTH * rows
   *   used_keep_last: the effective keep_last applied (may be 0 on fallback)
   */
  public function plan(int $keep_last = 0): array {
    if (!$this->fieldExists() || !$this->nodeFieldDataExists()) {
      return ['rows' => 0, 'approx_bytes' => 0, 'used_keep_last' => 0];
    }

    // Try information_schema first, then fall back to SHOW TABLE STATUS.
    $avgLen = 0;
    try {
      $schema = $this->db->getConnectionOptions()['database'] ?? '';
      $avgLen = (int) ($this->db->query(
        "SELECT AVG_ROW_LENGTH
         FROM information_schema.TABLES
         WHERE TABLE_SCHEMA = :s
           AND TABLE_NAME = 'node_revision__layout_builder__layout'",
        [':s' => $schema]
      )->fetchField() ?: 0);
    } catch (\Throwable $e) {}
    if ($avgLen === 0) {
      try {
        $row = $this->db->query("SHOW TABLE STATUS LIKE 'node_revision__layout_builder__layout'")->fetchObject();
        if ($row && isset($row->Avg_row_length)) {
          $avgLen = (int) $row->Avg_row_length;
        }
      } catch (\Throwable $e) {}
    }

    $keep_last = max(0, (int) $keep_last);
    $effectiveKeep = $keep_last;

    if ($keep_last > 0 && !$this->mysql8OrHigher()) {
      $effectiveKeep = 0;
    }

    if ($effectiveKeep === 0) {
      $rows = (int) $this->db->query("
        SELECT COUNT(*)
        FROM node_revision__layout_builder__layout r
        JOIN node_field_data nfd
          ON nfd.nid = r.entity_id AND nfd.langcode = r.langcode
        WHERE r.revision_id <> nfd.vid
      ")->fetchField();
    }
    else {
      $rows = (int) $this->db->query("
        WITH ranked AS (
          SELECT r.entity_id, r.langcode, r.revision_id,
                 ROW_NUMBER() OVER (PARTITION BY r.entity_id, r.langcode ORDER BY r.revision_id DESC) AS rn
          FROM node_revision__layout_builder__layout r
        )
        SELECT COUNT(*)
        FROM ranked rk
        JOIN node_field_data nfd
          ON nfd.nid = rk.entity_id AND nfd.langcode = rk.langcode
        WHERE rk.revision_id <> nfd.vid
          AND rk.rn > :keepN
      ", [':keepN' => $effectiveKeep])->fetchField();
    }

    return [
      'rows' => $rows,
      'approx_bytes' => $rows * $avgLen,
      'used_keep_last' => $effectiveKeep,
    ];
  }

  /**
   * EXECUTE: Delete stale Layout Builder revision rows in chunks.
   *
   * Safe: never touches node__layout_builder__layout.
   *
   * @param int $chunk
   *   Rows to delete per loop (default 5000).
   * @param int $keep_last
   *   Keep last N previous revisions per node/langcode (besides current).
   *   N>0 requires MySQL 8+. On older MySQL, execution falls back to N=0.
   *
   * @return array{deleted:int, used_keep_last:int}
   *   deleted: total rows deleted.
   *   used_keep_last: effective keep_last applied (may be 0 on fallback).
   */
  public function execute(int $chunk = 5000, int $keep_last = 0): array {
    if (!$this->fieldExists() || !$this->nodeFieldDataExists()) {
      return ['deleted' => 0, 'used_keep_last' => 0];
    }

    $chunk = max(1000, (int) $chunk);
    $keep_last = max(0, (int) $keep_last);
    $supportsWindows = $this->mysql8OrHigher();
    $effectiveKeep = ($keep_last > 0 && !$supportsWindows) ? 0 : $keep_last;

    $deleted = 0;

    if ($effectiveKeep === 0) {
      do {
        $this->dropTemp('tmp_lb_rev');
        $this->createTempTable(
          'tmp_lb_rev',
          '(entity_id INT UNSIGNED, langcode VARCHAR(12), revision_id INT UNSIGNED, KEY (revision_id))'
        );

        $this->db->query("
          INSERT INTO tmp_lb_rev (entity_id, langcode, revision_id)
          SELECT r.entity_id, r.langcode, r.revision_id
          FROM node_revision__layout_builder__layout r
          JOIN node_field_data nfd
            ON nfd.nid = r.entity_id AND nfd.langcode = r.langcode
          WHERE r.revision_id <> nfd.vid
          LIMIT {$chunk}
        ");

        $round = (int) $this->db->query("SELECT COUNT(*) FROM tmp_lb_rev")->fetchField();
        if ($round === 0) {
          $this->dropTemp('tmp_lb_rev');
          break;
        }

        $txn = $this->db->startTransaction();
        $this->db->query("
          DELETE r
          FROM node_revision__layout_builder__layout r
          JOIN tmp_lb_rev c
            ON c.entity_id = r.entity_id
           AND c.revision_id = r.revision_id
           AND c.langcode = r.langcode
        ");
        unset($txn);

        $deleted += $round;
        $this->dropTemp('tmp_lb_rev');
      } while ($round > 0);
    }
    else {
      // Window-function path (MySQL 8+).
      do {
        $this->dropTemp('tmp_lb_cand');
        $this->createTempTable(
          'tmp_lb_cand',
          '(entity_id INT UNSIGNED, langcode VARCHAR(12), revision_id INT UNSIGNED, KEY (revision_id))'
        );

        // Fill candidates: non-current rows excluding the last N per node/langcode.
        $this->db->query("
          INSERT INTO tmp_lb_cand (entity_id, langcode, revision_id)
          WITH ranked AS (
            SELECT r.entity_id, r.langcode, r.revision_id,
                   ROW_NUMBER() OVER (PARTITION BY r.entity_id, r.langcode ORDER BY r.revision_id DESC) AS rn
            FROM node_revision__layout_builder__layout r
          )
          SELECT rk.entity_id, rk.langcode, rk.revision_id
          FROM ranked rk
          JOIN node_field_data nfd
            ON nfd.nid = rk.entity_id AND nfd.langcode = rk.langcode
          WHERE rk.revision_id <> nfd.vid
            AND rk.rn > :keepN
          LIMIT {$chunk}
        ", [':keepN' => $effectiveKeep]);

        $round = (int) $this->db->query("SELECT COUNT(*) FROM tmp_lb_cand")->fetchField();
        if ($round === 0) {
          $this->dropTemp('tmp_lb_cand');
          break;
        }

        $txn = $this->db->startTransaction();
        $this->db->query("
          DELETE r
          FROM node_revision__layout_builder__layout r
          JOIN tmp_lb_cand c
            ON c.entity_id = r.entity_id
           AND c.revision_id = r.revision_id
           AND c.langcode = r.langcode
        ");
        unset($txn);

        $deleted += $round;
        $this->dropTemp('tmp_lb_cand');
      } while ($round > 0);
    }

    $this->logger->notice('Layout Builder revision cleanup deleted @count rows (keep_last=@keep).', [
      '@count' => number_format($deleted),
      '@keep' => $effectiveKeep,
    ]);

    return ['deleted' => $deleted, 'used_keep_last' => $effectiveKeep];
  }

  /**
   * Create a TEMPORARY table, prefer ENGINE=MEMORY, fall back gracefully.
   */
  private function createTempTable(string $name, string $columnsSql): void {
    try {
      $this->db->query("CREATE TEMPORARY TABLE {$name} {$columnsSql} ENGINE=MEMORY");
    } catch (\Throwable $e) {
      // Fallback without ENGINE clause.
      $this->db->query("CREATE TEMPORARY TABLE {$name} {$columnsSql}");
    }
  }

  /**
   * Drop a TEMPORARY table if it exists.
   */
  private function dropTemp(string $name): void {
    try {
      $this->db->query("DROP TEMPORARY TABLE IF EXISTS {$name}");
    } catch (\Throwable $e) {
    }
  }

}
