<?php

namespace Drupal\fast_revision_purge\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Psr\Log\LoggerInterface;

/**
 * Discovers revision tables and ERR (entity_reference_revisions) field tables.
 *
 * Notes:
 * - We collect *revision field* tables, not base field tables.
 *   e.g. node:         node_revision__field_foo
 *        paragraphs:   paragraph_revision__field_foo
 * - ERR tables are a subset of the above where the target_type === 'paragraph'.
 *   These are the edges used for graph-walking (BFS) between paragraph revisions.
 * - Paragraphs are included only if the module is enabled and the tables exist.
 * - D10/D11 differences:
 *   * Core revision table names can differ (e.g., `paragraphs_item_revision`),
 *     but revision FIELD tables are consistently `paragraph_revision__<field>`.
 *
 * Enhancement:
 * - We also union any `${entity}_revision__%` tables discovered via INFORMATION_SCHEMA
 *   to catch orphaned per-field revision tables that no longer have storage defs.
 */
final class RevisionTableMap {

  /**
   * Constructs a new RevisionTableMap.
   *
   * @param \Drupal\Core\Database\Connection $db
   *   DB connection for schema introspection.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $efm
   *   Field manager to enumerate field storage definitions.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   To check if the paragraphs module is enabled.
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger for diagnostics.
   */
  public function __construct(
    private Connection $db,
    private EntityFieldManagerInterface $efm,
    private ModuleHandlerInterface $moduleHandler,
    private LoggerInterface $logger,
  ) {}

  /**
   * In-memory memoized table map.
   *
   * @var array<string,mixed>|null
   */
  private ?array $cache = null;

  /**
   * Build (and memoize) the table map.
   *
   * @return array{
   *   node: array{field_tables:string[], err_tables:string[]},
   *   paragraph: array{field_tables:string[], err_tables:string[]}
   * }
   */
  public function build(): array {
    if ($this->cache !== null) {
      return $this->cache;
    }

    $schema = $this->db->schema();

    $map = [
      'node' => [
        'field_tables' => [],
        'err_tables' => [],
      ],
      'paragraph' => [
        'field_tables' => [],
        'err_tables' => [],
      ],
    ];

    // --- Node revisionable fields (from EFM) ---
    foreach ($this->efm->getFieldStorageDefinitions('node') as $storage) {
      if (!$storage->isRevisionable()) {
        continue;
      }
      $table = 'node_revision__' . $storage->getName();
      if ($schema->tableExists($table)) {
        $map['node']['field_tables'][] = $table;

        // ERR pointing to paragraphs become edges in our paragraph graph.
        if ($storage->getType() === 'entity_reference_revisions') {
          $settings = $storage->getSettings();
          if (!empty($settings['target_type']) && $settings['target_type'] === 'paragraph') {
            $map['node']['err_tables'][] = $table;
          }
        }
      }
    }

    // --- Paragraphs revisionable fields (from EFM, only if module enabled) ---
    if ($this->moduleHandler->moduleExists('paragraphs')) {
      foreach ($this->efm->getFieldStorageDefinitions('paragraph') as $storage) {
        if (!$storage->isRevisionable()) {
          continue;
        }
        // Field tables are consistently prefixed 'paragraph_revision__'.
        $table = 'paragraph_revision__' . $storage->getName();
        if ($schema->tableExists($table)) {
          $map['paragraph']['field_tables'][] = $table;

          // ERR to paragraphs enables nested paragraph → paragraph traversal.
          if ($storage->getType() === 'entity_reference_revisions') {
            $settings = $storage->getSettings();
            if (!empty($settings['target_type']) && $settings['target_type'] === 'paragraph') {
              $map['paragraph']['err_tables'][] = $table;
            }
          }
        }
      }
    }
    else {
      $this->logger->debug('FastRev: paragraphs module not enabled; paragraph tables omitted.');
    }

    // --- Enhancement: union with INFORMATION_SCHEMA for orphan tables ---
    $dbName = (string) ($this->db->getConnectionOptions()['database'] ?? '');
    if ($dbName !== '') {
      // Any node per-field revision tables.
      $nodeExtra = $this->listTablesLike($dbName, 'node\\_revision\\__%');
      // Any paragraph per-field revision tables.
      $paraExtra = $this->listTablesLike($dbName, 'paragraph\\_revision\\__%');
      // Merge + dedupe.
      $map['node']['field_tables'] = $this->uniqueStrings(array_merge($map['node']['field_tables'], $nodeExtra));
      $map['paragraph']['field_tables'] = $this->uniqueStrings(array_merge($map['paragraph']['field_tables'], $paraExtra));
    }

    return $this->cache = $map;
  }

  /**
   * Node revision field tables that are ERR → paragraph (first-hop edges).
   *
   * @return string[]
   */
  public function getNodeErrParagraphTables(): array {
    return $this->build()['node']['err_tables'];
  }

  /**
   * Paragraph revision field tables that are ERR → paragraph (nested edges).
   *
   * @return string[]
   */
  public function getParagraphErrParagraphTables(): array {
    return $this->build()['paragraph']['err_tables'];
  }

  /**
   * All node revision field tables (includes non-ERR revisionable fields).
   *
   * @return string[]
   */
  public function getNodeRevisionFieldTables(): array {
    return $this->build()['node']['field_tables'];
  }

  /**
   * All paragraph revision field tables (includes non-ERR revisionable fields).
   *
   * @return string[]
   */
  public function getParagraphRevisionFieldTables(): array {
    return $this->build()['paragraph']['field_tables'];
  }

  /**
   * Convenience: return the Layout Builder node revision field table name, if present.
   *
   * @return string|null
   *   The LB revision field table name, or NULL if not found.
   */
  public function getLayoutBuilderRevisionFieldTableName(): ?string {
    foreach ($this->getNodeRevisionFieldTables() as $t) {
      // Exact match first.
      if ($t === 'node_revision__layout_builder__layout') {
        return $t;
      }
      // Defensive: table prefixes or alternative naming that ends with the canonical tail.
      if (str_ends_with($t, '__layout_builder__layout')) {
        return $t;
      }
    }
    return NULL;
  }

  // ---------------- helpers ----------------

  /**
   * List tables matching a LIKE pattern (ESCAPE '\\').
   *
   * @param string $schemaName
   * @param string $like
   * @return string[]
   */
  private function listTablesLike(string $schemaName, string $like): array {
    try {
      $result = $this->db->query("
        SELECT TABLE_NAME
        FROM information_schema.TABLES
        WHERE TABLE_SCHEMA = :s AND TABLE_NAME LIKE :p ESCAPE '\\\\'
      ", [':s' => $schemaName, ':p' => $like])->fetchCol();
      // Sanitize to simple identifiers only.
      $out = [];
      foreach ((array) $result as $t) {
        if (is_string($t) && preg_match('/^[A-Za-z0-9_]+$/', $t)) {
          $out[] = $t;
        }
      }
      return $out;
    } catch (\Throwable $e) {
      $this->logger->warning('FastRev: listTablesLike failed for pattern @p: @m', ['@p' => $like, '@m' => $e->getMessage()]);
      return [];
    }
  }

  /**
   * Unique, preserve order.
   *
   * @param string[] $items
   * @return string[]
   */
  private function uniqueStrings(array $items): array {
    $seen = [];
    $out = [];
    foreach ($items as $t) {
      if (!isset($seen[$t])) {
        $seen[$t] = true;
        $out[] = $t;
      }
    }
    return $out;
  }

}
