<?php

namespace Drupal\fast_revision_purge\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Schema;
use Psr\Log\LoggerInterface;

/**
 * Ensures helpful DB indexes for fast planning/purging.
 *
 * Idempotent: safe to call multiple times; existing indexes are skipped.
 * Supports both Paragraphs schemas:
 *  - Newer: paragraph / paragraph_revision / paragraph_field_revision
 *  - Older: paragraphs_item / paragraphs_item_revision / paragraphs_item_field_data
 */
final class IndexManager {

  /**
   * Constructs the IndexManager service.
   *
   * @param \Drupal\Core\Database\Connection $db
   *   Database connection used for schema operations.
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger for diagnostics (index creation messages).
   * @param \Drupal\fast_revision_purge\Service\RevisionTableMap $map
   *   Table discovery helper (node/paragraph revision & field tables).
   */
  public function __construct(
    private Connection $db,
    private LoggerInterface $logger,
    private RevisionTableMap $map,
  ) {}

  /**
   * Build table map and ensure all recommended indexes exist.
   */
  public function ensureHelpfulIndexes(): void {
    $this->ensureIndexes($this->map->build());
  }

  /**
   * Ensure indexes on core and field revision tables.
   *
   * @param array $map
   *   Shape:
   *   - node:
   *     - core.revision: string (e.g., 'node_revision')
   *     - core.field_revision: string (e.g., 'node_field_revision')
   *     - field_tables: string[] (e.g., 'node_revision__field_foo', ...)
   *     - err_tables: string[] (subset of field_tables with ERR to paragraphs)
   *   - paragraph:
   *     - core.revision: string ('paragraph_revision' or 'paragraphs_item_revision')
   *     - core.field_revision: string ('paragraph_field_revision' or 'paragraphs_item_revision_field_data')
   *     - field_tables: string[] (e.g., 'paragraph_revision__field_foo', 'paragraph_r__hash', ...)
   *     - err_tables: string[] (subset with ERR to paragraphs for nesting)
   */
  public function ensureIndexes(array $map): void {
    // --- Node core tables ---
    $this->safeAddIndex('node_revision', 'fastrev_nrev_nid_ts_vid', ['nid', 'revision_timestamp', 'vid']);

    $this->safeAddIndex('node_field_revision', 'fastrev_nfr_vid', ['vid']);
    $this->safeAddIndex('node_field_revision', 'fastrev_nfr_nid_lang_vid', ['nid', 'langcode', 'vid']);

    // Node revision field tables, including ERR tables to paragraphs.
    foreach ($map['node']['field_tables'] as $t) {
      $this->safeAddIndex($t, 'fastrev_rev_id', ['revision_id']);
    }
    foreach ($map['node']['err_tables'] as $t) {
      $col = $this->errTargetRevisionColumn($t);
      $this->safeAddIndex($t, 'fastrev_target_rev', [$col]);
    }

    // --- Paragraphs core tables (support both schemas) ---
    $schema = $this->db->schema();
    $rev = '';
    $revData = '';
    $base = '';
    $data = '';
    try {
      $def = \Drupal::entityTypeManager()->getDefinition('paragraph', FALSE);
      if ($def) {
        $base = (string) ($def->get('base_table') ?? '');
        $data = (string) ($def->get('data_table') ?? '');
        $rev = (string) ($def->get('revision_table') ?? '');
        $revData = (string) ($def->get('revision_data_table') ?? '');
      }
    }
    catch (\Throwable $e) {
      // Fall through to presence checks below.
    }

    // Normalize to actual existing tables.
    $rev = $schema->tableExists($rev) ? $rev
      : ($schema->tableExists('paragraphs_item_revision') ? 'paragraphs_item_revision' : '');

    $revData = $schema->tableExists($revData) ? $revData
      : ($schema->tableExists('paragraphs_item_revision_field_data') ? 'paragraphs_item_revision_field_data' : '');

    // For protecting current pointers in Planner, these are helpful too.
    $base = $schema->tableExists($base) ? $base
      : ($schema->tableExists('paragraphs_item') ? 'paragraphs_item' : '');

    $data = $schema->tableExists($data) ? $data
      : ($schema->tableExists('paragraphs_item_field_data') ? 'paragraphs_item_field_data' : '');

    // Indexes for paragraph *revision* table.
    if ($rev) {
      $tsCol = '';
      foreach (['revision_timestamp', 'revision_created', 'created', 'changed'] as $c) {
        if ($schema->fieldExists($rev, $c)) { $tsCol = $c; break; }
      }
      if ($tsCol) {
        $this->safeAddIndex($rev, 'fastrev_prev_id_ts_rid', ['id', $tsCol, 'revision_id']);
      }
      else {
        $this->safeAddIndex($rev, 'fastrev_prev_id_rid', ['id', 'revision_id']);
      }
      $this->safeAddIndex($rev, 'fastrev_prev_rid', ['revision_id']);
    }

    // Indexes for paragraph *revision data* table.
    if ($revData) {
      $this->safeAddIndex($revData, 'fastrev_pfr_rid', ['revision_id']);
    }

    // Helpful pointer indexes on base/data (used when protecting current revs).
    if ($base && $schema->fieldExists($base, 'revision_id')) {
      $this->safeAddIndex($base, 'fastrev_pbase_rev', ['revision_id']);
    }
    if ($data && $schema->fieldExists($data, 'revision_id')) {
      $this->safeAddIndex($data, 'fastrev_pdata_rev', ['revision_id']);
    }

    // Paragraph revision field tables (map already resolves both patterns).
    foreach ($map['paragraph']['field_tables'] as $t) {
      $this->safeAddIndex($t, 'fastrev_rev_id', ['revision_id']);
    }
    foreach ($map['paragraph']['err_tables'] as $t) {
      $col = $this->errTargetRevisionColumn($t);
      $this->safeAddIndex($t, 'fastrev_target_rev', [$col]);
    }
  }

  /**
   * Resolve the ERR target revision column from a revision field table name.
   *
   * For tables like 'node_revision__field_foo', the ERR column name is expected
   * to be 'field_foo_target_revision_id'. If the table name isn't in the expected
   * double-underscore format, fall back to the generic 'target_revision_id'.
   *
   * @param string $revTable
   *   The revision field table name.
   *
   * @return string
   *   The column name that stores the target paragraph revision ID.
   */
  private function errTargetRevisionColumn(string $revTable): string {
    $pos = strpos($revTable, '__');
    if ($pos === false) {
      return 'target_revision_id';
    }
    $field = substr($revTable, $pos + 2);
    return $field . '_target_revision_id';
  }

  /**
   * Safely add an index if the table exists and the index is missing.
   *
   * Any exceptions during schema ops are logged as warnings and swallowed,
   * so this can run in environments with partial schemas without breaking.
   *
   * @param string $table
   *   Table name to index.
   * @param string $name
   *   Index machine name to create.
   * @param string[] $fields
   *   Column(s) to include in the index.
   */
  private function safeAddIndex(string $table, string $name, array $fields): void {
    $schema = $this->db->schema();
    try {
      if (!$schema->tableExists($table)) {
        return;
      }
      if ($schema->indexExists($table, $name)) {
        return;
      }
      $this->addIndexCompat($schema, $table, $name, $fields);
      $this->logger->debug('FastRev: added index @idx on @tbl', ['@idx' => $name, '@tbl' => $table]);
    }
    catch (\Throwable $e) {
      $this->logger->warning('FastRev: could not add index @idx on @tbl: @msg', [
        '@idx' => $name, '@tbl' => $table, '@msg' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Adds an index across DB drivers with differing signatures.
   */
  private function addIndexCompat(Schema $schema, string $table, string $index_name, array $columns): void {
    try {
      // Contrib mysql driver expects 4th $spec arg.
      $schema->addIndex($table, $index_name, $columns, []);
    }
    catch (\ArgumentCountError|\TypeError $e) {
      // Core driver (and older signatures) accept the 3-arg form.
      $schema->addIndex($table, $index_name, $columns);
    }
  }

}
