<?php

namespace Drupal\fast_revision_purge\Batch;

use Drupal\Core\Messenger\MessengerInterface;

/**
 * Batch callbacks that execute the actual purge in controlled chunks.
 *
 * Reads chunk/sleep from the batch context, invokes the Purger service, and
 * writes deletion totals back to the context so the UI can display them.
 */
final class PurgeBatch {

  /**
   * Processes the purge in one batch operation.
   *
   * Reads `chunk` and `sleep` from `$context['results']`, invokes the Purger
   * service, and writes deletion totals back to `$context['results']`.
   *
   * Expected context input:
   * - results.chunk (int): Number of revisions to delete per iteration.
   * - results.sleep (int): Sleep time in milliseconds between iterations.
   *
   * Populates on completion:
   * - results.node_deleted (int): Total node revisions deleted in this run.
   * - results.para_deleted (int): Total paragraph revisions deleted in this run.
   * - results.lb_rows_deleted (int): Rows deleted from LB rev field table.
   * - results.bytes_freed (int): Estimated bytes freed in this run.
   *
   * @param array $context
   *   Batch context array, passed by reference.
   */
  public static function process(array &$context): void {
    $context['results'] = $context['results'] ?? [];
    $chunk = (int) ($context['results']['chunk'] ?? 5000);
    $sleep = (int) ($context['results']['sleep'] ?? 0);

    $context['message'] = \Drupal::translation()->translate('Purging revisions...');

    /** @var \Drupal\fast_revision_purge\Service\Purger $purger */
    $purger = \Drupal::service('fast_revision_purge.purger');
    $result = $purger->purge((int) $chunk, (int) $sleep);

    // Basic deletion totals.
    $context['results']['node_deleted'] = $result->nodeRevisionsDeleted;
    $context['results']['para_deleted'] = $result->paragraphRevisionsDeleted;
    $context['results']['lb_rows_deleted'] = $result->layoutBuilderRowsDeleted;

    // Bytes freed are calculated & persisted by Purger::purge() into fastrev_stats.
    // Read it back so the UI can show it immediately.
    /** @var \Drupal\fast_revision_purge\Service\StatsStorage $stats */
    $stats = \Drupal::service('fast_revision_purge.stats');
    $row = $stats->get();
    $context['results']['bytes_freed'] = (int) ($row['space_freed_last_run'] ?? 0);

    $context['finished'] = 1;
    $context['message'] = \Drupal::translation()->translate('Purge complete.');
  }

  /**
   * Batch finished callback (no-op).
   *
   * @param bool $success
   *   TRUE if the batch completed successfully.
   * @param array $results
   *   Collected results (unused; see $context['results'] instead).
   * @param array $operations
   *   Remaining operations if aborted (unused).
   */
  public static function finished(bool $success, array $results, array $operations): void {
    $messenger = \Drupal::messenger();

    if ($success) {
      $messenger->addStatus(t('Purge completed successfully. To reclaim disk space, run ANALYZE/OPTIMIZE on large revision tables after hours. See the "Post-purge maintenance" section on this page for ready-to-copy SQL.'));
    }
    else {
      $messenger->addError(t('The purge batch ended with errors. Check the logs for details.'));
    }
  }

}
