<?php

namespace Drupal\fast_revision_purge\Batch;

/**
 * Batch callbacks for computing the Fast Revision Purge plan.
 *
 * Runs the Planner in a batch-safe context and relies on the Planner
 * to persist any dry-run stats (e.g., potential_claimable_space and
 * last_dryrun_timestamp) to fastrev_stats.
 */
final class PlanBatch {

  /**
   * Executes the planning step (single operation; Planner handles all phases).
   *
   * Options array mirrors UI/Drush arguments:
   * - keep_last (int): Keep latest N non-default node revisions per node.
   * - since (string|null): Date string (YYYY-MM-DD) to keep revisions since.
   * - protect_published (bool): Keep latest published revision per node.
   * - per_language (bool): Partition keep_last by (nid, langcode).
   * - keep_par_last (int): Keep last M paragraph revisions per paragraph.
   *
   * @param array $opts
   *   Planning options (see above).
   * @param array $context
   *   Batch context array (message/progress can be set here).
   */
  public static function phase(array $opts, array &$context): void {
    /** @var \Drupal\fast_revision_purge\Service\Planner $planner */
    $planner = \Drupal::service('fast_revision_purge.planner');

    $planner->plan(
      (int) ($opts['keep_last'] ?? 0),
      $opts['since'] ?? null,
      !empty($opts['protect_published']),
      !empty($opts['per_language']),
      (int) ($opts['keep_par_last'] ?? 0)
    );

    $context['message'] = \Drupal::translation()->translate('FastRev planning complete.');
    $context['finished'] = 1;
  }

  /**
   * Batch finished callback (no-op).
   *
   * @param bool $success
   *   TRUE if the batch completed successfully.
   * @param array $results
   *   Collected results (unused).
   * @param array $operations
   *   Remaining operations if aborted (unused).
   */
  public static function finish(bool $success, array $results, array $operations): void {
  }

}
