<?php

namespace Drupal\fast_revision_purge\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Component\Utility\Html;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Datetime\DateFormatterInterface;

use Drupal\fast_revision_purge\Service\Planner;
use Drupal\fast_revision_purge\Service\Purger;
use Drupal\fast_revision_purge\Service\IndexManager;
use Drupal\fast_revision_purge\Service\TableStats;
use Drupal\fast_revision_purge\Service\LayoutBuilderRevisionTruncator;
use Drupal\fast_revision_purge\Service\ParagraphRevisionTruncator;
use Drupal\fast_revision_purge\Service\StatsStorage;

/**
 * Admin UI for Fast Revision Purge.
 *
 * Provides configuration, a batch-based dry run (plan), an optional
 * "sanity query" helper, and one-click purge launchers.
 */
class SettingsForm extends ConfigFormBase {

  /** Core + module services */
  protected ModuleHandlerInterface $moduleHandler;
  protected MessengerInterface $messengerService;
  protected DateFormatterInterface $dateFormatter;

  /** Fast Revision Purge services */
  protected Planner $planner;
  protected Purger $purger;
  protected IndexManager $indexManager;
  protected TableStats $tableStats;
  protected StatsStorage $stats;

  /** DB + special truncators */
  protected Connection $db;
  protected LayoutBuilderRevisionTruncator $lbTruncator;
  protected ParagraphRevisionTruncator $paragraphTruncator;

  /**
   * Construct the settings form with injected services.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Config factory required by ConfigFormBase.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   Module handler for checking enabled modules.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   Messenger service for UI feedback.
   * @param \Drupal\fast_revision_purge\Service\Planner $planner
   *   Planner service that computes keep/delete sets.
   * @param \Drupal\fast_revision_purge\Service\Purger $purger
   *   Purger service that executes chunked deletes.
   * @param \Drupal\fast_revision_purge\Service\IndexManager $indexManager
   *   Ensures DB indexes that speed up planning/purging.
   * @param \Drupal\fast_revision_purge\Service\TableStats $tableStats
   *   Helper for database/table size calculations.
   * @param \Drupal\fast_revision_purge\Service\StatsStorage $stats
   *   Storage for last dry-run/purge stats.
   * @param \Drupal\Core\Database\Connection $db
   *   Database connection used for summary queries.
   * @param \Drupal\fast_revision_purge\Service\LayoutBuilderRevisionTruncator $lbTruncator
   *   Safe truncator for Layout Builder field revisions.
   * @param \Drupal\fast_revision_purge\Service\ParagraphRevisionTruncator $paragraphTruncator
   *   Safe truncator for Paragraph revisions.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   Date formatter to show relative times.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ModuleHandlerInterface $module_handler,
    MessengerInterface $messenger,
    Planner $planner,
    Purger $purger,
    IndexManager $indexManager,
    TableStats $tableStats,
    StatsStorage $stats,
    Connection $db,
    LayoutBuilderRevisionTruncator $lbTruncator,
    ParagraphRevisionTruncator $paragraphTruncator,
    DateFormatterInterface $dateFormatter,
  ) {
    parent::__construct($config_factory);
    $this->moduleHandler = $module_handler;
    $this->messengerService = $messenger;
    $this->planner = $planner;
    $this->purger = $purger;
    $this->indexManager = $indexManager;
    $this->tableStats = $tableStats;
    $this->stats = $stats;
    $this->db = $db;
    $this->lbTruncator = $lbTruncator;
    $this->paragraphTruncator = $paragraphTruncator;
    $this->dateFormatter = $dateFormatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('module_handler'),
      $container->get('messenger'),
      $container->get('fast_revision_purge.planner'),
      $container->get('fast_revision_purge.purger'),
      $container->get('fast_revision_purge.index_manager'),
      $container->get('fast_revision_purge.table_stats'),
      $container->get('fast_revision_purge.stats'),
      $container->get('database'),
      $container->get('fast_revision_purge.lb_truncator'),
      $container->get('fast_revision_purge.paragraph_truncator'),
      $container->get('date.formatter'),
    );
  }

  /**
   * Convenience accessor for typed messenger in this form.
   *
   * @return \Drupal\Core\Messenger\MessengerInterface
   *   The messenger service.
   */
  public function messenger(): MessengerInterface {
    return $this->messengerService;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'fast_revision_purge_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['fast_revision_purge.settings'];
  }

  /**
   * {@inheritdoc}
   *
   * Builds retention policy controls, execution tuning, danger zone,
   * DB overview (with top tables), and Plan/Purge actions.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);
    $config = $this->config('fast_revision_purge.settings');

    // --------------------
    // Database overview
    // --------------------
    $form['overview'] = [
      '#type' => 'details',
      '#title' => $this->t('Database overview'),
      '#open' => TRUE,
      '#cache' => ['max-age' => 0],
    ];

    // Current DB Size.
    $dbSize = TableStats::humanBytes($this->tableStats->getDatabaseSizeBytes());
    $form['overview']['db_size'] = [
      '#type' => 'item',
      '#title' => $this->t('Current DB Size'),
      '#markup' => '<pre>' . $dbSize . '</pre>',
    ];

    // Top 3 biggest tables (one per line).
    $lines = [];
    foreach ($this->tableStats->getTopTables(3) as $t) {
      $lines[] = sprintf('%s — %s', Html::escape($t['name']), TableStats::humanBytes($t['bytes']));
    }
    $form['overview']['top_tables'] = [
      '#type' => 'item',
      '#title' => $this->t('Top 3 biggest tables'),
      '#markup' => '<pre>' . implode("\n", $lines) . '</pre>',
    ];

    // Last Run (respect NULL = never).
    $s = $this->stats->get();
    $lastDryTs = $s['last_dryrun_timestamp'] ?? NULL;
    $lastPurgeTs = $s['last_purge_timestamp'] ?? NULL;

    $dryAgo = isset($lastDryTs) ? $this->dateFormatter->formatTimeDiffSince((int) $lastDryTs) : (string) $this->t('never');
    $purgeAgo = isset($lastPurgeTs) ? $this->dateFormatter->formatTimeDiffSince((int) $lastPurgeTs) : (string) $this->t('never');

    $form['overview']['last_run'] = [
      '#type' => 'item',
      '#title' => $this->t('Last Run'),
      '#markup' => '<pre>' .
        $this->t('Last Dry Run: @dry', ['@dry' => $dryAgo]) . "\n" .
        $this->t('Last Purged: @purge', ['@purge' => $purgeAgo]) .
      '</pre>',
    ];

    // Potential space that can be cleaned up (from stats, updated by Planner).
    $potentialBytesRaw = $s['potential_claimable_space'] ?? NULL;
    $potentialDisplay = isset($potentialBytesRaw) ? TableStats::humanBytes((int) $potentialBytesRaw) : (string) $this->t('N/A');
    $form['overview']['potential_cleanup'] = [
      '#type' => 'item',
      '#title' => $this->t('Potential space that can be cleaned up'),
      '#markup' => '<pre>' . $potentialDisplay . '</pre>',
    ];

    // --------------------
    // Retention policy
    // --------------------
    $form['policy'] = [
      '#type' => 'details',
      '#title' => $this->t('Retention policy'),
      '#open' => TRUE,
    ];

    $form['policy']['keep_last'] = [
      '#type' => 'number',
      '#title' => $this->t('Keep latest N non-default node revisions'),
      '#default_value' => $config->get('keep_last') ?? 5,
      '#min' => 0,
      '#required' => TRUE,
    ];

    $form['policy']['per_language'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Apply keep-last per language'),
      '#default_value' => (bool) ($config->get('per_language') ?? FALSE),
    ];

    $form['policy']['since'] = [
      '#type' => 'date',
      '#title' => $this->t('Keep revisions since date'),
      '#default_value' => $config->get('since') ?: NULL,
      '#description' => $this->t('Format: YYYY-MM-DD. Leave empty to disable.'),
    ];

    $form['policy']['protect_published'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Protect latest published revision per node'),
      '#default_value' => (bool) ($config->get('protect_published') ?? TRUE),
    ];

    $form['policy']['keep_paragraph_last'] = [
      '#type' => 'number',
      '#title' => $this->t('Keep last M paragraph revisions per paragraph entity'),
      '#default_value' => $config->get('keep_paragraph_last') ?? 1,
      '#min' => 0,
      '#required' => TRUE,
    ];

    // --------------------
    // Execution settings
    // --------------------
    $form['execution'] = [
      '#type' => 'details',
      '#title' => $this->t('Execution settings'),
      '#open' => TRUE,
    ];

    $form['execution']['chunk_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Chunk size'),
      '#default_value' => $config->get('chunk_size') ?? 5000,
      '#min' => 100,
      '#description' => $this->t('Delete this many revisions per transaction.'),
    ];

    $form['execution']['sleep_ms'] = [
      '#type' => 'number',
      '#title' => $this->t('Sleep between chunks (ms)'),
      '#default_value' => $config->get('sleep_ms') ?? 0,
      '#min' => 0,
      '#description' => $this->t('Small delay to reduce lock pressure.'),
    ];

    $form['execution']['ensure_indexes'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Ensure helpful DB indexes before running'),
      '#default_value' => FALSE,
    ];

    // --------------------
    // Sanity checks
    // --------------------
    $form['sanity'] = [
      '#type' => 'details',
      '#title' => $this->t('Sanity checks'),
      '#open' => TRUE,
    ];
    $form['sanity']['show_sanity_queries'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show sanity check queries (copy/paste)'),
      '#default_value' => FALSE,
      '#description' => $this->t('Outputs a Drush eval and raw SQL, pre-filled with your actual table names.'),
    ];

    // --------------------
    // Danger zone
    // --------------------
    $form['danger'] = [
      '#type' => 'details',
      '#title' => $this->t('Danger zone'),
      '#open' => TRUE,
      '#weight' => 97,
    ];

    $form['danger']['confirm_purge'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('I understand this will permanently delete revisions.'),
      '#default_value' => FALSE,
    ];

    // -------------------------------------------------
    // Optional Purge for Paragraphs and Layout builder.
    // -------------------------------------------------
    $hasParagraphs = $this->moduleHandler->moduleExists('paragraphs');
    $hasLayoutBuilder = $this->moduleHandler->moduleExists('layout_builder');

    $form['extra_purges'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['extra-purges']],
      '#weight' => 98,
    ];

    $form['extra_purges']['purge_paragraph_revisions'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Purge Paragraph Revisions'),
      '#default_value' => 0,
      '#disabled' => !$hasParagraphs,
      '#description' => !$hasParagraphs
        ? $this->t('Disabled: the Paragraphs module is not enabled.')
        : $this->t('Deletes non-current paragraph revision rows safely (keeps current).'),
    ];

    $form['extra_purges']['purge_layout_builder_revisions'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Purge Layout Builder Revisions'),
      '#default_value' => 0,
      '#disabled' => !$hasLayoutBuilder,
      '#description' => !$hasLayoutBuilder
        ? $this->t('Disabled: the Layout Builder module is not enabled.')
        : $this->t('Deletes non-current Layout Builder field revision rows safely (keeps current).'),
    ];

    // --------------------
    // Actions
    // --------------------
    $form['actions'] = ['#type' => 'actions', '#weight' => 99];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save configuration'),
      '#button_type' => 'primary',
    ];
    $form['actions']['dry_run'] = [
      '#type' => 'submit',
      '#value' => $this->t('Plan (Dry run)'),
      '#submit' => ['::submitDryRun'],
    ];
    $form['actions']['purge'] = [
      '#type' => 'submit',
      '#value' => $this->t('Run purge now'),
      '#submit' => ['::submitPurgeBatch'],
      '#attributes' => ['class' => ['button--danger']],
    ];

    // ---------------------------
    // Dry-run report (guarded)
    // ---------------------------

    // Snapshot counts (same logic as before) so we can decide whether to render.
    $report = $form_state->get('fastrev_report');
    if (!$report) {
      $schema = $this->db->schema();
      $report = [
        'node_keep'   => 0,
        'node_delete' => 0,
        'par_use'     => 0,
        'par_delete'  => 0,
        'lb_keep'     => 0,
        'lb_delete'   => 0,
        'sample_node' => [],
        'sample_par'  => [],
      ];
      if ($schema->tableExists('fastrev_node_keep')) {
        $report['node_keep'] = (int) $this->db->query("SELECT COUNT(*) FROM fastrev_node_keep")->fetchField();
      }
      if ($schema->tableExists('fastrev_node_delete')) {
        $report['node_delete'] = (int) $this->db->query("SELECT COUNT(*) FROM fastrev_node_delete")->fetchField();
        $report['sample_node'] = $this->db->query("SELECT vid FROM fastrev_node_delete ORDER BY vid LIMIT 10")->fetchCol();
      }
      if ($schema->tableExists('fastrev_par_in_use')) {
        $report['par_use'] = (int) $this->db->query("SELECT COUNT(*) FROM fastrev_par_in_use")->fetchField();
      }
      if ($schema->tableExists('fastrev_par_delete')) {
        $report['par_delete'] = (int) $this->db->query("SELECT COUNT(*) FROM fastrev_par_delete")->fetchField();
        $report['sample_par']  = $this->db->query("SELECT rid FROM fastrev_par_delete ORDER BY rid LIMIT 10")->fetchCol();
      }
      if ($schema->tableExists('fastrev_lb_keep')) {
        $report['lb_keep'] = (int) $this->db->query("SELECT COUNT(*) FROM fastrev_lb_keep")->fetchField();
      }
      if ($schema->tableExists('fastrev_lb_delete')) {
        $report['lb_delete'] = (int) $this->db->query("SELECT COUNT(*) FROM fastrev_lb_delete")->fetchField();
      }
    }

    // Only show the report if we have actually executed a dry run
    $has_dryrun = isset($lastDryTs);
    $total_found = (int) $report['node_keep']
      + (int) $report['node_delete']
      + (int) $report['par_use']
      + (int) $report['par_delete']
      + (int) $report['lb_keep']
      + (int) $report['lb_delete'];

    if ($has_dryrun || $total_found > 0) {
      $form['report'] = [
        '#type' => 'details',
        '#title' => $this->t('Dry run report'),
        '#open' => TRUE,
        '#cache' => ['max-age' => 0],
      ];

      $items = [];
      $items[] = $this->t('Node revisions: KEEP=@k, DELETE=@d', [
        '@k' => $report['node_keep'],
        '@d' => $report['node_delete'],
      ]);
      $items[] = $this->t('Paragraph revisions: KEEP=@k, DELETE=@d', [
        '@k' => $report['par_use'],
        '@d' => $report['par_delete'],
      ]);
      $items[] = $this->t('Layout Builder revisions: KEEP=@k, DELETE=@d', [
        '@k' => $report['lb_keep'],
        '@d' => $report['lb_delete'],
      ]);

      $form['report']['summary'] = [
        '#theme' => 'item_list',
        '#items' => $items,
      ];

      if (!empty($report['sample_node']) || !empty($report['sample_par'])) {
        $sitems = [];
        if (!empty($report['sample_node'])) {
          $sitems[] = $this->t('Sample node vids to delete: @ids', ['@ids' => implode(', ', $report['sample_node'])]);
        }
        if (!empty($report['sample_par'])) {
          $sitems[] = $this->t('Sample paragraph rids to delete: @ids', ['@ids' => implode(', ', $report['sample_par'])]);
        }
        $form['report']['samples'] = ['#theme' => 'item_list', '#items' => $sitems];
      }
    }
    else {
      $form['report_empty'] = [
        '#type' => 'item',
        '#title' => $this->t('Dry run report'),
        '#markup' => '<em>' . $this->t('No dry run has been executed yet.') . '</em>',
      ];
    }

    if ($form_state->getValue('show_sanity_queries')) {
      $q = $this->buildSanityQueries();
      $form['sanity_output'] = [
        '#type' => 'details',
        '#title' => $this->t('Sanity check queries (copy/paste)'),
        '#open' => TRUE,
      ];
      $form['sanity_output']['drush'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Drush eval'),
        '#default_value' => $q['drush'],
        '#rows' => 12,
      ];
      $form['sanity_output']['sql'] = [
        '#type' => 'textarea',
        '#title' => $this->t('SQL (MySQL)'),
        '#default_value' => $q['sql'],
        '#rows' => 10,
      ];
    }

    // ----------------------------
    // Post-purge maintenance (SQL)
    // ----------------------------
    if (isset($lastPurgeTs)) {
      $form['post_maintenance'] = [
        '#type' => 'details',
        '#title' => $this->t('Post-purge maintenance (optional, recommended after hours)'),
        '#open' => FALSE,
        '#weight' => 100,
        '#description' => $this->t('MySQL does not shrink files on DELETE. To reclaim space on disk (if per-table tablespaces are enabled), run these commands during a maintenance window. Backups recommended.'),
      ];

      $form['post_maintenance']['sql'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Copy-paste SQL'),
        '#rows' => 14,
        '#default_value' => $this->buildPostPurgeSql(),
        '#description' => $this->t('Run in a MySQL client with appropriate privileges.'),
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $since = $form_state->getValue('since');
    if ($since && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $since)) {
      $form_state->setErrorByName('since', $this->t('Date must be in YYYY-MM-DD format.'));
    }
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->saveConfigFromValues($form_state);
    $this->messenger()->addStatus($this->t('Configuration saved.'));
    parent::submitForm($form, $form_state);
  }

  /**
   * Submit handler for the "Plan (Dry run)" button.
   *
   * @param array $form
   *   The form render array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitDryRun(array &$form, FormStateInterface $form_state): void {
    $this->saveConfigFromValues($form_state, FALSE);

    if ($form_state->getValue('ensure_indexes')) {
      $this->indexManager->ensureHelpfulIndexes();
      $this->messenger()->addStatus($this->t('Indexes ensured.'));
    }

    $cfg = $this->config('fast_revision_purge.settings');
    $opts = [
      'since'             => $cfg->get('since') ?: null,
      'protect_published' => (bool) $cfg->get('protect_published'),
      'keep_last'         => (int) $cfg->get('keep_last'),
      'per_language'      => (bool) $cfg->get('per_language'),
      'keep_par_last'     => (int) $cfg->get('keep_paragraph_last'),
    ];

    $batch = [
      'title' => $this->t('Planning (dry run)…'),
      'operations' => [
        [[ '\\Drupal\\fast_revision_purge\\Batch\\PlanBatch', 'phase' ], [$opts]],
      ],
      'finished' => [ '\\Drupal\\fast_revision_purge\\Batch\\PlanBatch', 'finish' ],
      'progress_message' => $this->t('Planning…'),
      'error_message' => $this->t('An error occurred during planning.'),
    ];

    batch_set($batch);
    $this->messenger()->addStatus($this->t('Dry run started. Report will appear when the batch completes.'));
    $form_state->setRebuild();
  }

  /**
   * Submit handler for the "Run purge now" button.
   *
   * @param array $form
   *   The form render array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitPurgeBatch(array &$form, FormStateInterface $form_state): void {
    if (!$form_state->getValue('confirm_purge')) {
      $this->messenger()->addError($this->t('Please confirm purge by checking the box in the Danger zone.'));
      return;
    }

    $this->saveConfigFromValues($form_state);

    if ($form_state->getValue('ensure_indexes')) {
      $this->indexManager->ensureHelpfulIndexes();
      $this->messenger()->addStatus($this->t('Indexes ensured.'));
    }

    $cfg   = $this->config('fast_revision_purge.settings');
    $chunk = (int) ($cfg->get('chunk_size') ?? 5000);
    $sleep = (int) ($cfg->get('sleep_ms') ?? 0);

    $hasParagraphs    = $this->moduleHandler->moduleExists('paragraphs');
    $hasLayoutBuilder = $this->moduleHandler->moduleExists('layout_builder');

    $operations = [];

    if ($form_state->getValue('purge_paragraph_revisions') && $hasParagraphs) {
      $operations[] = [
        [static::class, 'opParagraphPurge'],
        [['chunk' => $chunk]],
      ];
    }

    if ($form_state->getValue('purge_layout_builder_revisions') && $hasLayoutBuilder) {
      $operations[] = [
        [static::class, 'opLayoutBuilderPurge'],
        [['chunk' => $chunk, 'lb_keep_last' => 0]],
      ];
    }

    $operations[] = [
      ['\\Drupal\\fast_revision_purge\\Batch\\PurgeBatch', 'process'],
      [],
    ];

    $batch = [
      'title' => $this->t('Purging revisions'),
      'operations' => $operations,
      'finished' => ['\\Drupal\\fast_revision_purge\\Batch\\PurgeBatch', 'finished'],
      'init_message' => $this->t('Initializing purge…'),
      'progress_message' => $this->t('Working…'),
      'error_message' => $this->t('An error occurred during purge.'),
      'results' => ['chunk' => $chunk, 'sleep' => $sleep],
    ];

    batch_set($batch);
  }

  /**
   * Persist form values, then snapshot report counts/samples into form state.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param bool $snapshotReport
   *   Whether to compute and store a counts snapshot in $form_state.
   */
  private function saveConfigFromValues(FormStateInterface $form_state, bool $snapshotReport = TRUE): void {
    $this->configFactory->getEditable('fast_revision_purge.settings')
      ->set('keep_last', (int) $form_state->getValue('keep_last'))
      ->set('since', $form_state->getValue('since') ?: NULL)
      ->set('protect_published', (bool) $form_state->getValue('protect_published'))
      ->set('keep_paragraph_last', (int) $form_state->getValue('keep_paragraph_last'))
      ->set('per_language', (bool) $form_state->getValue('per_language'))
      ->set('chunk_size', (int) $form_state->getValue('chunk_size'))
      ->set('sleep_ms', (int) $form_state->getValue('sleep_ms'))
      ->save();

    if (!$snapshotReport) {
      return;
    }

    $schema = $this->db->schema();
    $report = [
      'node_keep'   => 0,
      'node_delete' => 0,
      'par_use'     => 0,
      'par_delete'  => 0,
      'lb_keep'     => 0,
      'lb_delete'   => 0,
      'sample_node' => [],
      'sample_par'  => [],
    ];

    if ($schema->tableExists('fastrev_node_keep')) {
      $report['node_keep'] = (int) $this->db->query("SELECT COUNT(*) FROM fastrev_node_keep")->fetchField();
    }
    if ($schema->tableExists('fastrev_node_delete')) {
      $report['node_delete'] = (int) $this->db->query("SELECT COUNT(*) FROM fastrev_node_delete")->fetchField();
      $report['sample_node'] = $this->db->query("SELECT vid FROM fastrev_node_delete ORDER BY vid LIMIT 10")->fetchCol();
    }
    if ($schema->tableExists('fastrev_par_in_use')) {
      $report['par_use'] = (int) $this->db->query("SELECT COUNT(*) FROM fastrev_par_in_use")->fetchField();
    }
    if ($schema->tableExists('fastrev_par_delete')) {
      $report['par_delete'] = (int) $this->db->query("SELECT COUNT(*) FROM fastrev_par_delete")->fetchField();
      $report['sample_par'] = $this->db->query("SELECT rid FROM fastrev_par_delete ORDER BY rid LIMIT 10")->fetchCol();
    }
    if ($schema->tableExists('fastrev_lb_keep')) {
      $report['lb_keep'] = (int) $this->db->query("SELECT COUNT(*) FROM fastrev_lb_keep")->fetchField();
    }
    if ($schema->tableExists('fastrev_lb_delete')) {
      $report['lb_delete'] = (int) $this->db->query("SELECT COUNT(*) FROM fastrev_lb_delete")->fetchField();
    }

    $form_state->set('fastrev_report', $report);
  }

  /**
   * Build copy-paste sanity-check queries using actual table names.
   *
   * @return array{drush:string,sql:string}
   *   An array with 'drush' and 'sql' keys containing query text.
   */
  private function buildSanityQueries(): array {
    $def = \Drupal::entityTypeManager()->getDefinition('paragraph', FALSE);
    $base = $def ? (string) ($def->get('base_table') ?? '') : '';
    $data = $def ? (string) ($def->get('data_table') ?? '') : '';
    $rev  = $def ? (string) ($def->get('revision_table') ?? '') : '';
    $baseOrData = $base ?: $data;

    $schema = \Drupal::database()->schema();
    $tsCol = '';
    if ($rev && $schema->tableExists($rev)) {
      foreach (['revision_timestamp', 'revision_created', 'created', 'changed'] as $c) {
        if ($schema->fieldExists($rev, $c)) { $tsCol = $c; break; }
      }
    }

    // Drush eval.
    $drush = "// 1) Default paragraph revisions must NOT be in delete set.\n";
    $drush .= "\$base = '" . addslashes($baseOrData) . "';\n";
    $drush .= "if (\$base) {\n";
    $drush .= "  \$n = (int) \\Drupal::database()->query(\"SELECT COUNT(*) FROM fastrev_par_delete d JOIN {$baseOrData} p ON p.revision_id = d.rid\")->fetchField();\n";
    $drush .= "  echo \"Default paragraph revs in delete set: \$n\\n\";\n";
    $drush .= "}\n\n";
    $drush .= "// 2) Sample 20 paragraph deletions with ordering by detected timestamp (or revision_id fallback).\n";
    $drush .= "\$rev = '" . addslashes($rev) . "';\n";
    $drush .= "\$ts = '" . addslashes($tsCol) . "';\n";
    $drush .= "if (\$rev) {\n";
    $drush .= "  if (\$ts) {\n";
    $drush .= "    \$sql = \"SELECT d.rid, pr.id, pr.`\$ts` AS ts FROM fastrev_par_delete d JOIN {\$rev} pr ON pr.revision_id = d.rid ORDER BY pr.`\$ts` DESC, pr.revision_id DESC LIMIT 20\";\n";
    $drush .= "  } else {\n";
    $drush .= "    \$sql = \"SELECT d.rid, pr.id FROM fastrev_par_delete d JOIN {\$rev} pr ON pr.revision_id = d.rid ORDER BY pr.revision_id DESC LIMIT 20\";\n";
    $drush .= "  }\n";
    $drush .= "  foreach (\\Drupal::database()->query(\$sql)->fetchAll() as \$row) { print_r(\$row); }\n";
    $drush .= "}\n";

    // Raw SQL.
    $sql = "";
    if (!empty($baseOrData)) {
      $sql .= "/* Default paragraph revisions MUST NOT be in delete set (expect 0) */\n";
      $sql .= "SELECT COUNT(*) AS should_be_zero\n";
      $sql .= "FROM fastrev_par_delete d\nJOIN `{$baseOrData}` p ON p.revision_id = d.rid;\n\n";
    }
    if (!empty($rev)) {
      $sql .= "/* Sample 20 paragraph revisions slated for deletion */\n";
      if ($tsCol) {
        $sql .= "SELECT d.rid, pr.id, pr.`{$tsCol}` AS ts\n";
        $sql .= "FROM fastrev_par_delete d\nJOIN `{$rev}` pr ON pr.revision_id = d.rid\n";
        $sql .= "ORDER BY pr.`{$tsCol}` DESC, pr.revision_id DESC\nLIMIT 20;\n";
      }
      else {
        $sql .= "SELECT d.rid, pr.id\n";
        $sql .= "FROM fastrev_par_delete d\nJOIN `{$rev}` pr ON pr.revision_id = d.rid\n";
        $sql .= "ORDER BY pr.revision_id DESC\nLIMIT 20;\n";
      }
    }

    return ['drush' => $drush, 'sql' => $sql];
  }

  /**
   * Batch op: Purge non-current paragraph revisions.
   *
   * @param array $options
   *   Options array. Supports:
   *   - chunk: (int) number of rows per iteration, minimum 1000.
   * @param array $context
   *   Batch context (by reference).
   */
  public static function opParagraphPurge(array $options, &$context): void {
    $chunk = isset($options['chunk']) ? max(1000, (int) $options['chunk']) : 5000;
    $context['message'] = \Drupal::translation()->translate('Purging paragraph revisions...');

    /** @var \Drupal\fast_revision_purge\Service\ParagraphRevisionTruncator $svc */
    $svc = \Drupal::service('fast_revision_purge.paragraph_truncator');
    $res = $svc->execute($chunk);
    $tick = (int) ($res['deleted'] ?? 0);
    $context['results']['extras']['paragraphs_deleted'] =
      (int) ($context['results']['extras']['paragraphs_deleted'] ?? 0) + $tick;

    if ($tick > 0) {
      $context['finished'] = 0;
      $context['message'] = \Drupal::translation()->translate(
        'Purging paragraph revisions… deleted @n so far.',
        ['@n' => number_format($context['results']['extras']['paragraphs_deleted'])]
      );
    } else {
      $context['finished'] = 1;
      $context['message'] = \Drupal::translation()->translate('Paragraph revision purge complete.');
    }
  }

  /**
   * Batch op: Purge non-current Layout Builder field revisions.
   *
   * @param array $options
   *   Options array. Supports:
   *   - chunk: (int) number of rows per iteration, minimum 1000.
   *   - lb_keep_last: (int) number of most-recent LB rows to keep per entity.
   * @param array $context
   *   Batch context (by reference).
   */
  public static function opLayoutBuilderPurge(array $options, &$context): void {
    $chunk = isset($options['chunk']) ? max(1000, (int) $options['chunk']) : 5000;
    $keep  = isset($options['lb_keep_last']) ? max(0, (int) $options['lb_keep_last']) : 0;
    $context['message'] = \Drupal::translation()->translate('Purging Layout Builder revisions…');

    /** @var \Drupal\fast_revision_purge\Service\LayoutBuilderRevisionTruncator $svc */
    $svc = \Drupal::service('fast_revision_purge.lb_truncator');
    $res = $svc->execute($chunk, $keep);
    $tick = (int) ($res['deleted'] ?? 0);
    $context['results']['extras']['layout_builder_deleted'] =
      (int) ($context['results']['extras']['layout_builder_deleted'] ?? 0) + $tick;
    $context['results']['extras']['layout_builder_keep_last'] =
      (int) ($res['used_keep_last'] ?? $keep);

    if ($tick > 0) {
      $context['finished'] = 0;
      $context['message'] = \Drupal::translation()->translate(
        'Purging Layout Builder revisions… deleted @n so far.',
        ['@n' => number_format($context['results']['extras']['layout_builder_deleted'])]
      );
    } else {
      $context['finished'] = 1;
      $context['message'] = \Drupal::translation()->translate('Layout Builder purge complete.');
    }
  }
  
  /**
   * Build ANALYZE/OPTIMIZE SQL for likely-impacted revision tables.
   *
   * @return string
   *   SQL text to copy/paste after a purge.
   */
  private function buildPostPurgeSql(): string {
    $schema = $this->db->schema();

    // Always consider these core revision tables.
    $must = [
      'node_revision',
      'node_field_revision',
      'node_revision__layout_builder__layout',
    ];

    // Augment with the largest node revision field tables we can see.
    // We’ll scan the top 12 tables and keep those that start with 'node_revision__'.
    $candidates = [];
    foreach ($this->tableStats->getTopTables(12) as $t) {
      $name = (string) $t['name'];
      if (str_starts_with($name, 'node_revision__')) {
        $candidates[] = $name;
      }
    }

    // Merge, de-dupe, and keep only tables that actually exist.
    $tables = array_values(array_unique(array_merge($must, $candidates)));
    $tables = array_values(array_filter($tables, fn($t) => $schema->tableExists($t)));

    // Build SQL.
    $lines = [];
    $lines[] = "/* Check if per-table tablespaces are enabled (1 = shrink .ibd on OPTIMIZE) */";
    $lines[] = "SHOW VARIABLES LIKE 'innodb_file_per_table';";
    $lines[] = "";
    $lines[] = "/* Recommended: run after hours; requires free disk space ≈ table size for rebuild */";

    // ANALYZE (safe, quick)
    foreach ($tables as $t) {
      $lines[] = "ANALYZE TABLE `{$t}`;";
    }
    $lines[] = "";

    // OPTIMIZE (rebuilds table; reclaims space if file_per_table=ON)
    foreach ($tables as $t) {
      $lines[] = "OPTIMIZE TABLE `{$t}`;";
    }

    return implode("\n", $lines);
  }

}
